#!/usr/bin/env python3
"""
Pool Client API

Simple client interface for pool operations.
"""

from typing import Optional, Any
from contextlib import asynccontextmanager

from otpylib import gen_server, atom
from otpylib_pool.atoms import (
    CHECKOUT, CHECKIN, STATUS, WHICH_WORKERS, PING,
    OK, ERROR, TIMEOUT,
)


class Pool:
    """
    Simple client API for worker pool operations.
    
    Provides synchronous-style methods for interacting with a pool manager.
    """
    
    @staticmethod
    async def checkout(pool_name: str | atom.Atom, timeout: Optional[float] = 5.0) -> Any:
        """
        Checkout a worker from the pool.
        
        Args:
            pool_name: Name of the pool (registered name)
            timeout: Max time to wait for a worker (None = wait forever)
        
        Returns:
            Worker PID on success
            
        Raises:
            TimeoutError: If no worker available within timeout
            RuntimeError: If checkout fails for other reasons
        """
        result = await gen_server.call(pool_name, (CHECKOUT, timeout), timeout=timeout)
        
        match result:
            case (atom_ok, worker_pid) if atom_ok == OK:
                return worker_pid
            case (atom_timeout, _) if atom_timeout == TIMEOUT:
                raise TimeoutError(f"No workers available in pool '{pool_name}' within {timeout}s")
            case (atom_error, error) if atom_error == ERROR:
                raise RuntimeError(f"Checkout failed: {error}")
            case _:
                raise RuntimeError(f"Unexpected checkout response: {result}")
    
    @staticmethod
    async def checkin(pool_name: str | atom.Atom, worker_pid: Any) -> bool:
        """
        Check in a worker back to the pool.
        
        Args:
            pool_name: Name of the pool
            worker_pid: PID of the worker to return
            
        Returns:
            True on success
            
        Raises:
            RuntimeError: If checkin fails
        """
        result = await gen_server.call(pool_name, (CHECKIN, worker_pid), timeout=5.0)
        
        match result:
            case atom_ok if atom_ok == OK:
                return True
            case (atom_error, error) if atom_error == ERROR:
                raise RuntimeError(f"Checkin failed: {error}")
            case _:
                raise RuntimeError(f"Unexpected checkin response: {result}")
    
    @staticmethod
    async def status(pool_name: str | atom.Atom):
        """
        Get pool status information.
        
        Args:
            pool_name: Name of the pool
            
        Returns:
            PoolStatus object with pool statistics
        """
        result = await gen_server.call(pool_name, STATUS, timeout=5.0)
        
        match result:
            case (atom_error, error) if atom_error == ERROR:
                raise RuntimeError(f"Status query failed: {error}")
            case _:
                return result
    
    @staticmethod
    async def which_workers(pool_name: str | atom.Atom):
        """
        List all workers in the pool.
        
        Args:
            pool_name: Name of the pool
            
        Returns:
            List of worker info dictionaries
        """
        result = await gen_server.call(pool_name, WHICH_WORKERS, timeout=5.0)
        
        match result:
            case (atom_error, error) if atom_error == ERROR:
                raise RuntimeError(f"Worker list query failed: {error}")
            case _:
                return result
    
    @staticmethod
    async def ping(pool_name: str | atom.Atom) -> str:
        """
        Ping the pool manager for health check.
        
        Args:
            pool_name: Name of the pool
            
        Returns:
            "pong" on success
        """
        return await gen_server.call(pool_name, PING, timeout=5.0)
    
    @staticmethod
    @asynccontextmanager
    async def transaction(pool_name: str | atom.Atom, timeout: Optional[float] = 5.0):
        """
        Context manager for safe worker checkout/checkin.
        
        Automatically checks in the worker when the context exits,
        even if an exception occurs.
        
        Example:
            ```python
            async with Pool.transaction("db_pool") as worker_pid:
                result = await gen_server.call(worker_pid, ("query", "SELECT ..."))
            # Worker automatically returned to pool
            ```
        
        Args:
            pool_name: Name of the pool
            timeout: Max time to wait for checkout
            
        Yields:
            Worker PID
        """
        worker_pid = None
        try:
            worker_pid = await Pool.checkout(pool_name, timeout)
            yield worker_pid
        finally:
            if worker_pid is not None:
                try:
                    await Pool.checkin(pool_name, worker_pid)
                except Exception:
                    # Swallow checkin errors to avoid masking original exception
                    pass
