#!/usr/bin/env python3
"""
Pool Core Business Logic

Pure functions for pool management operations.
No side effects - all state transformations are explicit.
Returns Result types for error handling.
"""

import time
from typing import Optional, Tuple, List, Any
from result import Result, Ok, Err

from otpylib import dynamic_supervisor, gen_server, process
from otpylib.dynamic_supervisor import child_spec, options, ONE_FOR_ONE, PERMANENT
from otpylib.module import OTPModule, DYNAMIC_SUPERVISOR

from otpylib_pool.atoms import (
    WORKER_SUP, AVAILABLE, BUSY, OVERFLOW, FIFO, LIFO, OK, TIMEOUT,
)
from otpylib_pool.data import (
    PoolManagerState,
    WorkerInfo,
    CheckoutRequest,
    PoolStatus,
)


# =============================================================================
# Worker Supervisor Definition
# =============================================================================

class WorkerSupervisor(metaclass=OTPModule, behavior=DYNAMIC_SUPERVISOR, version="1.0.0"):
    """
    Dynamic supervisor for pool workers.
    
    Manages both fixed pool workers and overflow workers.
    """
    
    async def init(self, args):
        """Initialize with empty children - workers added dynamically."""
        children = []  # Start empty, workers added via start_child
        
        opts = options(
            strategy=ONE_FOR_ONE,
            max_restarts=10,
            max_seconds=60
        )
        
        return (children, opts)
    
    async def terminate(self, reason, state):
        """Called when supervisor terminates."""
        pass


# =============================================================================
# Supervisor Initialization
# =============================================================================

async def initialize_worker_supervisor(state: PoolManagerState) -> Result[Any, str]:
    """
    Initialize the dynamic supervisor for workers.
    
    Returns the supervisor PID on success.
    """
    try:
        # Generate unique name for this pool's worker supervisor
        sup_name = f"{state.pool_name}_worker_sup"
        
        sup_pid = await dynamic_supervisor.start_link(
            WorkerSupervisor,
            init_arg=None,
            name=sup_name
        )
        
        return Ok(sup_pid)
        
    except Exception as e:
        return Err(f"Failed to start worker supervisor: {e}")


# =============================================================================
# Worker Pool Initialization
# =============================================================================

async def start_initial_workers(state: PoolManagerState) -> Result[int, str]:
    """
    Start the initial fixed-size pool of workers.
    
    Returns the number of workers successfully started.
    """
    if not state.worker_supervisor_pid:
        return Err("Worker supervisor not initialized")
    
    created_count = 0
    
    for i in range(state.size):
        worker_id = f"{state.pool_name}_worker_{i}"
        
        spec = child_spec(
            id=worker_id,
            module=state.worker_module,
            args=state.worker_args,
            restart=PERMANENT,
            name=worker_id
        )
        
        try:
            success, result = await dynamic_supervisor.start_child(
                state.worker_supervisor_pid,
                spec
            )
            
            if success:
                worker_pid = result
                
                # Register worker in pool
                worker_info = WorkerInfo(
                    pid=worker_pid,
                    id=worker_id,
                    state=AVAILABLE,
                    is_overflow=False
                )
                state.workers[worker_pid] = worker_info
                state.available_workers.append(worker_pid)
                created_count += 1
                
                # Monitor worker for exits
                await process.monitor(worker_pid)
                
        except Exception as e:
            pass  # Continue trying to start other workers
    
    if created_count == 0:
        return Err("Failed to start any workers")
    
    return Ok(created_count)


# =============================================================================
# Worker Checkout
# =============================================================================

async def checkout_worker(from_pid: Any, timeout: Optional[float], state: PoolManagerState) -> Result[str, str]:
    """
    Checkout a worker from the pool.
    
    Returns:
        Ok(worker_pid) if worker available
        Err("queued") if no workers, request queued
        Err("timeout") if timeout and no workers
    """
    # Try to get an available worker
    if state.available_workers:
        # Get worker based on strategy
        if state.strategy == FIFO:
            worker_pid = state.available_workers.pop(0)  # First in, first out
        else:  # LIFO
            worker_pid = state.available_workers.pop()  # Last in, first out
        
        # Mark worker as busy
        worker_info = state.workers[worker_pid]
        worker_info.state = BUSY
        worker_info.checked_out_at = time.time()
        worker_info.checked_out_by = from_pid
        
        state.total_checkouts += 1
        
        return Ok(worker_pid)
    
    # No available workers - try overflow
    if state.current_overflow < state.max_overflow:
        result = await _create_overflow_worker(state)
        
        if result.is_ok():
            worker_pid = result.unwrap()
            
            # Mark as busy immediately
            worker_info = state.workers[worker_pid]
            worker_info.state = BUSY
            worker_info.checked_out_at = time.time()
            worker_info.checked_out_by = from_pid
            
            state.current_overflow += 1
            state.total_checkouts += 1
            
            return Ok(worker_pid)
    
    # No workers available and can't create overflow
    return Err("queued")


async def _create_overflow_worker(state: PoolManagerState) -> Result[Any, str]:
    """
    Create a temporary overflow worker.
    
    Returns the PID of the new worker if successful.
    """
    if not state.worker_supervisor_pid:
        return Err("Worker supervisor not initialized")
    
    # Overflow workers get unique IDs with timestamp
    worker_id = f"{state.pool_name}_overflow_{int(time.time() * 1000000)}"
    
    spec = child_spec(
        id=worker_id,
        module=state.worker_module,
        args=state.worker_args,
        restart=PERMANENT,
        name=worker_id
    )
    
    try:
        success, result = await dynamic_supervisor.start_child(
            state.worker_supervisor_pid,
            spec
        )
        
        if success:
            worker_pid = result
            
            # Register overflow worker
            worker_info = WorkerInfo(
                pid=worker_pid,
                id=worker_id,
                state=OVERFLOW,  # Mark as overflow initially
                is_overflow=True
            )
            state.workers[worker_pid] = worker_info
            
            # Monitor worker
            await process.monitor(worker_pid)
            
            return Ok(worker_pid)
            
    except Exception as e:
        return Err(f"Failed to create overflow worker: {e}")


# =============================================================================
# Worker Checkin
# =============================================================================

async def checkin_worker(worker_pid: Any, state: PoolManagerState) -> Result[bool, str]:
    """
    Check in a worker back to the pool.
    
    If there are waiting checkouts, immediately assign to next waiter.
    If overflow worker and no queue, terminate it.
    Otherwise return to available pool.
    """
    # Verify worker exists
    if worker_pid not in state.workers:
        return Err("Worker not found in pool")
    
    worker_info = state.workers[worker_pid]
    
    # Clear checkout info
    from_pid = worker_info.checked_out_by
    worker_info.checked_out_at = None
    worker_info.checked_out_by = None
    
    state.total_checkins += 1
    
    # Check if anyone is waiting
    if state.waiting_checkouts:
        checkout_req = state.waiting_checkouts.pop(0)
        
        # Assign worker to waiting caller
        worker_info.state = BUSY
        worker_info.checked_out_at = time.time()
        worker_info.checked_out_by = checkout_req.from_pid
        
        state.total_checkouts += 1
        
        # Reply to waiting caller
        await gen_server.reply(checkout_req.from_pid, (OK, worker_pid))
        
        return Ok(True)
    
    # No one waiting - handle based on worker type
    if worker_info.is_overflow:
        # Overflow worker - terminate it
        await _terminate_overflow_worker(worker_pid, state)
        return Ok(True)
    else:
        # Regular worker - return to pool
        worker_info.state = AVAILABLE
        state.available_workers.append(worker_pid)
        return Ok(True)


async def _terminate_overflow_worker(worker_pid: Any, state: PoolManagerState) -> Result[bool, str]:
    """
    Terminate an overflow worker and clean up state.
    """
    if worker_pid not in state.workers:
        return Err("Worker not found")
    
    worker_info = state.workers[worker_pid]
    
    try:
        # Remove from state first
        del state.workers[worker_pid]
        if state.current_overflow > 0:
            state.current_overflow -= 1
        
        # Terminate via supervisor
        await dynamic_supervisor.terminate_child(state.worker_supervisor_pid, worker_info.id)
        await dynamic_supervisor.delete_child(state.worker_supervisor_pid, worker_info.id)
        
        return Ok(True)
        
    except Exception as e:
        return Err(f"Failed to terminate overflow worker: {e}")


# =============================================================================
# Worker Exit Handling
# =============================================================================

async def handle_worker_exit(worker_pid: Any, reason: Any, state: PoolManagerState) -> Result[bool, str]:
    """
    Handle a worker process exit/crash.
    
    Removes worker from state and notifies waiting callers if needed.
    Supervisor will restart permanent workers automatically.
    """
    if worker_pid not in state.workers:
        # Unknown worker, ignore
        return Ok(True)
    
    worker_info = state.workers[worker_pid]
    
    # If worker was checked out, we need to handle the caller
    if worker_info.checked_out_by:
        caller_pid = worker_info.checked_out_by
        # Caller will get an error when they try to use the dead worker
        # Could optionally send them a message here
    
    # Remove from available list if present
    if worker_pid in state.available_workers:
        state.available_workers.remove(worker_pid)
    
    # Remove from state
    if worker_info.is_overflow and state.current_overflow > 0:
        state.current_overflow -= 1
    
    del state.workers[worker_pid]
    
    # Note: Supervisor will restart permanent workers automatically
    # We'll see them as new PIDs when they restart
    
    return Ok(True)


# =============================================================================
# Checkout Timeout Handling
# =============================================================================

async def handle_checkout_timeout(from_pid: Any, state: PoolManagerState) -> Result[bool, str]:
    """
    Handle a checkout request timeout.
    
    Remove request from queue and notify caller.
    """
    # Find and remove the request
    for i, req in enumerate(state.waiting_checkouts):
        if req.from_pid == from_pid:
            state.waiting_checkouts.pop(i)
            state.total_timeouts += 1
            
            # Reply with timeout
            await gen_server.reply(from_pid, (TIMEOUT, None))
            
            return Ok(True)
    
    # Request not found (may have been served already)
    return Ok(True)


# =============================================================================
# Status and Queries
# =============================================================================

async def get_pool_status(state: PoolManagerState) -> Result[PoolStatus, str]:
    """
    Build a status snapshot of the pool.
    """
    available_count = len(state.available_workers)
    busy_count = sum(1 for w in state.workers.values() if w.state == BUSY)
    overflow_count = sum(1 for w in state.workers.values() if w.is_overflow)
    waiting_count = len(state.waiting_checkouts)
    
    uptime = (time.time() - state.started_at.timestamp())
    
    status = PoolStatus(
        name=str(state.pool_name),
        size=state.size,
        max_overflow=state.max_overflow,
        strategy=state.strategy.name if hasattr(state.strategy, 'name') else str(state.strategy),
        available_workers=available_count,
        busy_workers=busy_count,
        overflow_workers=overflow_count,
        waiting_callers=waiting_count,
        total_checkouts=state.total_checkouts,
        total_checkins=state.total_checkins,
        total_timeouts=state.total_timeouts,
        uptime_seconds=uptime,
    )
    
    return Ok(status)


async def list_workers(state: PoolManagerState) -> Result[List[dict], str]:
    """
    List all workers with their current status.
    """
    workers = []
    
    for pid, info in state.workers.items():
        workers.append({
            'pid': pid,
            'id': info.id,
            'state': info.state.name if hasattr(info.state, 'name') else str(info.state),
            'is_overflow': info.is_overflow,
            'checked_out_by': info.checked_out_by,
            'checked_out_at': info.checked_out_at,
        })
    
    return Ok(workers)


# =============================================================================
# Cleanup
# =============================================================================

async def terminate_all_workers(state: PoolManagerState) -> Result[bool, str]:
    """
    Terminate all workers on pool shutdown.
    """
    if not state.worker_supervisor_pid:
        return Ok(True)
    
    try:
        # Supervisor shutdown will terminate all children
        # We just need to clear our state
        state.workers.clear()
        state.available_workers.clear()
        state.waiting_checkouts.clear()
        
        return Ok(True)
        
    except Exception as e:
        return Err(f"Failed to terminate workers: {e}")
