#!/usr/bin/env python3
"""
Pool Data Structures

Data structures and types for otpylib-pool.
Pure data - no business logic here.

Uses atoms from atoms.py for state discriminators and constants.
"""

from dataclasses import dataclass, field
from typing import Any, Optional, List, Dict
from datetime import datetime

from otpylib import atom
from otpylib_pool.atoms import (
    POOL_MANAGER,
    POOL_SUP,
    WORKER_SUP,
    AVAILABLE,
    BUSY,
    OVERFLOW,
    FIFO,
    LIFO,
)


# =============================================================================
# Worker Info
# =============================================================================

@dataclass
class WorkerInfo:
    """
    Information about a worker in the pool.
    
    State is represented as an atom (AVAILABLE, BUSY, OVERFLOW).
    """
    pid: Any
    id: str
    state: atom.Atom  # AVAILABLE, BUSY, or OVERFLOW
    checked_out_at: Optional[float] = None
    checked_out_by: Optional[Any] = None
    is_overflow: bool = False


@dataclass
class CheckoutRequest:
    """
    Represents a pending checkout request waiting for a worker.
    
    When no workers are available, requests are queued here.
    """
    from_pid: Any
    requested_at: float
    timeout: Optional[float] = None


# =============================================================================
# Pool State
# =============================================================================

@dataclass
class PoolManagerState:
    """
    State container for the Pool Manager GenServer.
    
    Tracks workers, checkout queue, and pool metrics.
    Strategy is stored as an atom (FIFO or LIFO).
    """
    started_at: datetime = field(default_factory=datetime.now)
    
    # Pool configuration
    pool_name: str | atom.Atom = ""
    worker_module: type = None
    worker_args: Any = None
    size: int = 5
    max_overflow: int = 0
    strategy: atom.Atom = FIFO  # FIFO or LIFO
    
    # Worker management
    workers: Dict[Any, WorkerInfo] = field(default_factory=dict)  # pid -> WorkerInfo
    available_workers: List[Any] = field(default_factory=list)  # List of available worker PIDs
    
    # Checkout queue
    waiting_checkouts: List[CheckoutRequest] = field(default_factory=list)
    
    # Metrics
    total_checkouts: int = 0
    total_checkins: int = 0
    total_timeouts: int = 0
    current_overflow: int = 0
    
    # Supervisor PID
    worker_supervisor_pid: Optional[Any] = None


# =============================================================================
# Pool Specification
# =============================================================================

@dataclass
class PoolSpec:
    """
    Specification for creating a worker pool.
    
    Args:
        name: Name to register the pool (atom or string)
        worker_module: The OTPModule class for workers (must be GEN_SERVER)
        worker_args: Arguments to pass to worker init
        size: Number of permanent workers in the pool
        max_overflow: Maximum number of overflow workers (0 = no overflow)
        strategy: Selection strategy atom (FIFO or LIFO)
        checkout_timeout: Default max time to wait for worker (seconds, None = infinite)
    """
    name: str | atom.Atom
    worker_module: type
    worker_args: Any = None
    size: int = 5
    max_overflow: int = 0
    strategy: atom.Atom = FIFO  # Can accept FIFO/LIFO atom
    checkout_timeout: Optional[float] = 5.0
    
    def __post_init__(self):
        """Validate and normalize the pool spec."""
        if not self.name:
            raise ValueError("Pool spec must have a name")
        
        if not self.worker_module:
            raise ValueError("Pool spec must have a worker_module")
        
        if self.size < 1:
            raise ValueError("Pool size must be at least 1")
        
        if self.max_overflow < 0:
            raise ValueError("Max overflow cannot be negative")
        
        # Normalize strategy to atom
        if isinstance(self.strategy, str):
            if self.strategy.lower() == "fifo":
                self.strategy = FIFO
            elif self.strategy.lower() == "lifo":
                self.strategy = LIFO
            else:
                raise ValueError(f"Invalid strategy: {self.strategy}. Must be 'fifo' or 'lifo'")


# =============================================================================
# Result Data Structures
# =============================================================================

@dataclass
class PoolStatus:
    """
    Status information about a pool.
    
    Returned by status queries to provide a snapshot of pool state.
    """
    name: str
    size: int
    max_overflow: int
    strategy: str  # "fifo" or "lifo"
    available_workers: int
    busy_workers: int
    overflow_workers: int
    waiting_callers: int
    total_checkouts: int
    total_checkins: int
    total_timeouts: int
    uptime_seconds: float
