#!/usr/bin/env python3
"""
Pool Manager Lifecycle

OTP supervision for the pool manager.
Uses the 0.5.0 OTPModule supervisor pattern.
"""

from otpylib import supervisor, gen_server
from otpylib.supervisor import PERMANENT, ONE_FOR_ONE
from otpylib.module import OTPModule, SUPERVISOR

from otpylib_pool.boundaries import PoolManager
from otpylib_pool.data import POOL_MANAGER, POOL_SUP, PoolSpec
from otpylib_pool.atoms import STOP


# =============================================================================
# Pool Supervisor
# =============================================================================

class PoolSupervisor(metaclass=OTPModule, behavior=SUPERVISOR, version="1.0.0"):
    """
    Pool Supervisor.
    
    Manages the PoolManager GenServer which in turn manages the worker
    supervisor and all pool workers.
    """
    
    async def init(self, pool_spec: PoolSpec):
        """Initialize the pool manager supervision tree."""
        # Create child spec for the pool manager GenServer
        pool_manager_name = f"{pool_spec.name}_manager"
        
        genserver_spec = supervisor.child_spec(
            id=pool_manager_name,
            module=PoolManager,
            args=pool_spec,
            restart=PERMANENT,
            name=pool_manager_name,
        )
        
        children = [genserver_spec]
        
        opts = supervisor.options(
            strategy=ONE_FOR_ONE,
            max_restarts=3,
            max_seconds=60
        )
        
        return (children, opts)
    
    async def terminate(self, reason, state):
        """Cleanup on termination."""
        pass


# =============================================================================
# Lifecycle Functions
# =============================================================================

async def start(pool_spec: PoolSpec):
    """
    Start the Pool Manager supervision tree.
    
    Args:
        pool_spec: PoolSpec configuration for the pool
    
    Returns:
        The PID of the spawned supervisor.
    """
    # Use the pool name for the supervisor registration
    sup_name = f"{pool_spec.name}_sup"
    
    return await supervisor.start_link(
        PoolSupervisor,
        init_arg=pool_spec,
        name=sup_name
    )


async def stop(pool_name: str):
    """
    Stop the Pool Manager gracefully.
    
    Args:
        pool_name: Name of the pool to stop
    """
    try:
        pool_manager_name = f"{pool_name}_manager"
        await gen_server.cast(pool_manager_name, STOP)
    except Exception:
        pass
