from functools import lru_cache
from typing import ClassVar

import torch
from PIL.Image import Image
from transformers import InstructBlipForConditionalGeneration, InstructBlipProcessor

from media_analyzer.machine_learning.caption.captioner_protocol import CaptionerProtocol


@lru_cache
def get_instructblip_assets() -> tuple[
    InstructBlipProcessor, InstructBlipForConditionalGeneration, str
]:
    """Retrieve and cache the InstructBLIP processor, model, and device.

    Loads the model in float16 for efficient inference.
    """
    device = "cuda" if torch.cuda.is_available() else "cpu"
    print(f"Loading InstructBLIP model on device: '{device}'")

    # We use the Flan-T5-XL based model.
    # torch_dtype=torch.float16 significantly reduces VRAM and increases speed.
    processor = InstructBlipProcessor.from_pretrained(  # type: ignore[no-untyped-call]
        "Salesforce/instructblip-flan-t5-xl"
    )
    model = InstructBlipForConditionalGeneration.from_pretrained(
        "Salesforce/instructblip-flan-t5-xl", torch_dtype=torch.float16
    ).to(device)
    model.eval()  # Set model to evaluation mode

    print("Model loaded successfully.")
    return processor, model, device


class InstructBlipCaptioner(CaptionerProtocol):
    """Captioner implementation using the InstructBLIP model."""

    hallucinated_words: ClassVar[list[str]] = ["arafed", "araffe"]

    def caption(self, image: Image, instruction: str | None = None) -> str:
        """Generate a caption for the given image based on an instruction.

        Args:
            image: The image to caption.
            instruction: An optional instruction to guide the generation.
                         If None, a default descriptive prompt is used.

        Returns:
            A formatted caption string.
        """
        # If no specific instruction is given, use a default one for general captioning.
        if instruction is None:
            instruction = "Provide a brief, one-sentence description of this image."

        raw_text = self.raw_caption(image, instruction)

        # Post-processing can be useful for any model
        processed_text = raw_text.replace(" ' ", "'")

        if all(word not in processed_text for word in self.hallucinated_words):
            return processed_text.capitalize()

        for fake_word in self.hallucinated_words:
            processed_text = processed_text.replace(fake_word, "")

        return processed_text.strip().capitalize()

    def raw_caption(self, image: Image, instruction: str) -> str:
        """Generate a raw caption for the image using the InstructBlip model.

        Args:
            image: The image to caption.
            instruction: The instruction to guide the model's output.

        Returns:
            The raw string generated by the model.
        """
        processor, model, device = get_instructblip_assets()
        inputs = processor(images=image.convert("RGB"), text=instruction, return_tensors="pt").to(
            device, dtype=torch.float16
        )

        outputs = model.generate(
            **inputs,
            do_sample=False,  # Stick with deterministic beam search for now
            num_beams=5,  # A slightly larger beam size can help find better sequences
            max_new_tokens=128,  # Keep the safety limit
            min_length=1,
            repetition_penalty=1.5,  # Prevents the model from repeating words
            length_penalty=1.2,  # Slightly encourages the model to complete its thoughts
        )

        caption = processor.batch_decode(  # type: ignore[no-untyped-call]
            outputs, skip_special_tokens=True
        )[0].strip()

        assert isinstance(caption, str)
        return caption
