"""当前上下文工具。

封装对当前协程内 Bot / Event 的访问，便于在 handler 中使用：

    from asa import ctx

    @on_group_message
    async def handle(event, bot):
        await ctx.reply("hello")  # 等价于 bot.reply(event, "hello")
"""

from __future__ import annotations

from dataclasses import dataclass
from typing import Any, Optional

from .bot import _current_bot, _current_event
from .event import Event


@dataclass
class _CtxProxy:
    """当前事件上下文的轻量代理。"""

    def get_bot(self) -> "Optional[Any]":
        try:
            return _current_bot.get()
        except LookupError:
            return None

    def get_event(self) -> Optional[Event]:
        try:
            return _current_event.get()
        except LookupError:
            return None

    # 便捷属性
    @property
    def bot(self) -> "Optional[Any]":
        return self.get_bot()

    @property
    def event(self) -> Optional[Event]:
        return self.get_event()

    # 便捷操作封装
    async def reply(self, message: str, **kwargs: Any) -> Any:
        """基于当前事件快速回复。

        等价于：

            bot = ctx.bot
            event = ctx.event
            await bot.reply(event, message, **kwargs)
        """
        bot = self.get_bot()
        event = self.get_event()
        if bot is None or event is None:
            raise RuntimeError("ctx.reply() 必须在事件处理上下文中调用")
        return await bot.reply(message, event, **kwargs)


ctx = _CtxProxy()


__all__ = ["ctx"]

