"""处理器自动发现工具。

默认行为：
    - 扫描并导入 `bot` 包下的所有子模块
    - 导入时触发装饰器执行，从而注册事件处理函数
"""

from __future__ import annotations

import importlib
import pkgutil
from types import ModuleType
from typing import Iterable


def _import_package(name: str) -> ModuleType | None:
    """安全导入一个包，不存在时返回 None。"""
    try:
        module = importlib.import_module(name)
    except ModuleNotFoundError:
        return None
    return module


def _walk_package(module: ModuleType) -> None:
    """导入包下所有子模块。"""
    if not hasattr(module, "__path__"):
        # 不是包（可能是单文件模块），直接返回
        return

    prefix = module.__name__ + "."
    for info in pkgutil.walk_packages(module.__path__, prefix):
        importlib.import_module(info.name)


def discover_handlers(packages: Iterable[str]) -> None:
    """导入并扫描给定包列表。

    Args:
        packages: 包名列表，例如 ["bot", "mybot.handlers"]
    """
    for name in packages:
        module = _import_package(name)
        if module is None:
            continue
        _walk_package(module)


__all__ = ["discover_handlers"]

