"""OneBot 11 消息事件视图。

目前主要兼容 OneBot 11 的私聊 / 群消息事件结构：
    https://github.com/botuniverse/onebot-11

本类只做一层很薄的封装，把原始事件里的常见字段暴露成属性，方便条件系统和业务代码使用。
"""

from __future__ import annotations

from dataclasses import dataclass
from typing import Any, Dict, Optional


@dataclass
class Event:
    """OneBot 11 消息事件视图。"""

    raw: Dict[str, Any]

    # ===== 通用基础字段 =====

    @property
    def time(self) -> Optional[int]:
        value = self.raw.get("time")
        try:
            return int(value) if value is not None else None
        except (TypeError, ValueError):
            return None

    @property
    def self_id(self) -> Optional[int]:
        value = self.raw.get("self_id")
        try:
            return int(value) if value is not None else None
        except (TypeError, ValueError):
            return None

    @property
    def post_type(self) -> Optional[str]:
        return self.raw.get("post_type")

    @property
    def message_type(self) -> Optional[str]:
        # "private" / "group"
        return self.raw.get("message_type")

    @property
    def sub_type(self) -> Optional[str]:
        # 私聊: friend/group/other；群聊: normal/anonymous/notice
        return self.raw.get("sub_type")

    @property
    def message_id(self) -> Optional[int]:
        value = self.raw.get("message_id")
        try:
            return int(value) if value is not None else None
        except (TypeError, ValueError):
            return None

    # ===== 发送方信息 =====

    @property
    def user_id(self) -> Optional[int]:
        """发送者 QQ 号。"""
        sender = self.raw.get("sender") or {}
        uid = sender.get("user_id") or self.raw.get("user_id")
        try:
            return int(uid) if uid is not None else None
        except (TypeError, ValueError):
            return None

    @property
    def group_id(self) -> Optional[int]:
        """群号，仅群消息有意义。"""
        gid = self.raw.get("group_id")
        try:
            return int(gid) if gid is not None else None
        except (TypeError, ValueError):
            return None

    @property
    def sender(self) -> Dict[str, Any]:
        """原始 sender 对象，可能为空字典。"""
        value = self.raw.get("sender")
        return value if isinstance(value, dict) else {}

    # ===== 匿名信息（群聊） =====

    @property
    def anonymous(self) -> Optional[Dict[str, Any]]:
        value = self.raw.get("anonymous")
        return value if isinstance(value, dict) else None

    # ===== 消息内容 =====

    @property
    def message(self) -> Any:
        """OneBot 11 的 message 字段（CQ 码或消息段结构）。"""
        return self.raw.get("message")

    @property
    def raw_message(self) -> Optional[str]:
        """原始消息内容字符串。"""
        msg = self.raw.get("raw_message") or self.raw.get("message")
        return msg if isinstance(msg, str) else None

    @property
    def font(self) -> Optional[int]:
        value = self.raw.get("font")
        try:
            return int(value) if value is not None else None
        except (TypeError, ValueError):
            return None

    @property
    def text(self) -> Optional[str]:
        """与 raw_message 等价的简写。"""
        return self.raw_message

    # ===== 便捷判断 =====

    @property
    def is_message(self) -> bool:
        return self.post_type == "message"

    @property
    def is_group(self) -> bool:
        return self.message_type == "group"

    @property
    def is_private(self) -> bool:
        return self.message_type == "private"

    @property
    def is_anonymous(self) -> bool:
        return self.sub_type == "anonymous"


__all__ = ["Event"]
