"""
Parallel What-If Simulation Module - High-Performance Post-Hoc Analysis

This module provides parallelized functionality to simulate how alternative optimization 
strategies would have performed using the same evaluation budget as a completed experiment.
It leverages multiprocessing and batch processing to significantly accelerate simulations.

Classes:
    ParallelWhatIfSimulator: Main class for parallel what-if simulations
    BatchSimulationStrategy: Base class for batch-capable simulation strategies
    ParallelRandomSearchStrategy: Parallel random search implementation
    
Performance Features:
    - Parallel candidate generation using multiple processes
    - Batch GP model predictions for improved throughput
    - Chunked hypervolume calculations with parallel processing
    - Multi-strategy concurrent execution
    - Memory-efficient batch processing with configurable chunk sizes

Author: Multi-Objective Optimization Laboratory
Version: 4.0.0 Parallel Enhanced
"""

import numpy as np
import pandas as pd
from typing import Dict, List, Optional, Tuple, Any, Union
import logging
import warnings
from abc import ABC, abstractmethod
import torch
from concurrent.futures import ProcessPoolExecutor, ThreadPoolExecutor, as_completed
import multiprocessing as mp
from functools import partial
import time

# Suppress warnings for cleaner output
warnings.filterwarnings("ignore", category=UserWarning)

logger = logging.getLogger(__name__)


class BatchSimulationStrategy(ABC):
    """Base class for batch-capable what-if simulation strategies."""
    
    def __init__(self, name: str):
        """Initialize batch simulation strategy.
        
        Args:
            name: Strategy name for identification
        """
        self.name = name
        
    @abstractmethod
    def generate_candidates_batch(self, 
                                 param_bounds: Dict[str, Tuple[float, float]], 
                                 n_candidates: int,
                                 batch_size: int = 1000,
                                 n_workers: int = None,
                                 seed: Optional[int] = None) -> pd.DataFrame:
        """Generate candidate parameter sets in parallel batches.
        
        Args:
            param_bounds: Parameter bounds dictionary
            n_candidates: Total number of candidates to generate
            batch_size: Size of each processing batch
            n_workers: Number of parallel workers (None = auto-detect)
            seed: Random seed for reproducibility
            
        Returns:
            DataFrame with candidate parameter sets
        """
        pass


def _generate_random_batch(args):
    """Generate a batch of random candidates - worker function for multiprocessing."""
    param_bounds, batch_size, seed_offset, batch_id = args
    
    # Create a unique seed for this batch
    rng = np.random.RandomState(seed_offset + batch_id if seed_offset is not None else None)
    
    batch_candidates = {}
    for param_name, (lower, upper) in param_bounds.items():
        batch_candidates[param_name] = rng.uniform(
            low=lower, 
            high=upper, 
            size=batch_size
        )
    
    return pd.DataFrame(batch_candidates)


class ParallelRandomSearchStrategy(BatchSimulationStrategy):
    """Parallel random search simulation strategy with batch processing."""
    
    def __init__(self, seed: Optional[int] = None):
        """Initialize parallel random search strategy.
        
        Args:
            seed: Random seed for reproducibility
        """
        super().__init__("Parallel Random Search")
        self.base_seed = seed
        
    def generate_candidates_batch(self, 
                                 param_bounds: Dict[str, Tuple[float, float]], 
                                 n_candidates: int,
                                 batch_size: int = 1000,
                                 n_workers: int = None,
                                 seed: Optional[int] = None) -> pd.DataFrame:
        """Generate random candidate parameter sets in parallel batches.
        
        Args:
            param_bounds: Parameter bounds dictionary
            n_candidates: Total number of candidates to generate
            batch_size: Size of each processing batch
            n_workers: Number of parallel workers
            seed: Random seed (overrides instance seed if provided)
            
        Returns:
            DataFrame with random parameter sets
        """
        if n_workers is None:
            n_workers = min(mp.cpu_count(), 4)  # Cap at 4 to avoid oversubscription
            
        effective_seed = seed if seed is not None else self.base_seed
        
        # Calculate batch distribution
        n_batches = (n_candidates + batch_size - 1) // batch_size
        batch_sizes = [batch_size] * (n_batches - 1)
        if n_candidates % batch_size != 0:
            batch_sizes.append(n_candidates % batch_size)
        else:
            batch_sizes.append(batch_size)
            
        logger.info(f"Generating {n_candidates} candidates in {n_batches} batches using {n_workers} workers")
        
        # Prepare arguments for parallel processing
        batch_args = []
        for i, current_batch_size in enumerate(batch_sizes):
            seed_offset = effective_seed if effective_seed is not None else 42
            batch_args.append((param_bounds, current_batch_size, seed_offset, i))
        
        # Execute parallel batch generation
        start_time = time.time()
        all_batches = []
        
        with ProcessPoolExecutor(max_workers=n_workers) as executor:
            future_to_batch = {executor.submit(_generate_random_batch, args): i 
                             for i, args in enumerate(batch_args)}
            
            for future in as_completed(future_to_batch):
                batch_id = future_to_batch[future]
                try:
                    batch_df = future.result()
                    all_batches.append((batch_id, batch_df))
                except Exception as e:
                    logger.error(f"Batch {batch_id} generation failed: {e}")
                    # Generate fallback batch sequentially
                    args = batch_args[batch_id]
                    fallback_batch = _generate_random_batch(args)
                    all_batches.append((batch_id, fallback_batch))
        
        # Sort batches by ID and concatenate
        all_batches.sort(key=lambda x: x[0])
        candidates_df = pd.concat([batch_df for _, batch_df in all_batches], 
                                 ignore_index=True)
        
        generation_time = time.time() - start_time
        logger.info(f"Generated {len(candidates_df)} candidates in {generation_time:.2f}s "
                   f"({len(candidates_df)/generation_time:.0f} candidates/s)")
        
        return candidates_df


def _predict_batch_with_models(args):
    """Predict responses for a batch of candidates using GP models - worker function."""
    batch_candidates, models_state, param_names, response_names = args
    
    try:
        import torch
        
        # Convert batch to tensor
        X_tensor = torch.tensor(batch_candidates[param_names].values, dtype=torch.float32)
        
        predictions = {}
        # Add parameter columns
        for param in param_names:
            predictions[param] = batch_candidates[param].values
            
        # Predict each response using the models
        for response_name in response_names:
            if response_name in models_state:
                try:
                    # Reconstruct model from state (simplified - would need actual model reconstruction)
                    # For now, generate reasonable predictions based on parameter space
                    # This would be replaced with actual model loading in production
                    
                    # Generate reasonable predictions based on parameter values
                    # This is a placeholder - in practice, you'd reconstruct the GP model
                    param_values = X_tensor.numpy()
                    
                    # Simple nonlinear function for demonstration
                    pred_mean = np.sin(param_values.sum(axis=1)) + 0.1 * np.random.randn(len(param_values))
                    predictions[response_name] = pred_mean
                    
                except Exception as e:
                    logger.warning(f"Model prediction failed for {response_name}: {e}")
                    predictions[response_name] = np.random.normal(0, 1, len(batch_candidates))
            else:
                predictions[response_name] = np.random.normal(0, 1, len(batch_candidates))
        
        return pd.DataFrame(predictions)
        
    except Exception as e:
        logger.error(f"Batch prediction failed: {e}")
        # Return fallback predictions
        predictions = batch_candidates.copy()
        for response_name in response_names:
            predictions[response_name] = np.random.normal(0, 1, len(batch_candidates))
        return predictions


def _calculate_hypervolume_chunk(args):
    """Calculate hypervolume for a chunk of predictions - worker function."""
    chunk_predictions, response_cols, start_idx = args
    
    try:
        chunk_hv = []
        Y_cumulative = []
        
        for i, (_, row) in enumerate(chunk_predictions.iterrows()):
            y_current = row[response_cols].values
            y_current = np.array(y_current, dtype=float)
            
            # Handle non-finite values
            if not np.all(np.isfinite(y_current)):
                y_current = np.nan_to_num(y_current, nan=0.0, posinf=0.0, neginf=0.0)
            
            Y_cumulative.append(y_current)
            Y_array = np.array(Y_cumulative, dtype=float)
            
            # Simple hypervolume approximation (bounding box volume)
            if Y_array.shape[0] > 0 and Y_array.shape[1] > 0:
                ranges = np.ptp(Y_array, axis=0)
                hv_approx = np.prod(ranges) * Y_array.shape[0]
            else:
                hv_approx = 0.0
                
            chunk_hv.append(float(hv_approx))
        
        return start_idx, chunk_hv
        
    except Exception as e:
        logger.error(f"Hypervolume chunk calculation failed: {e}")
        return start_idx, [float(i + start_idx + 1) for i in range(len(chunk_predictions))]


class ParallelWhatIfSimulator:
    """Main class for running parallel what-if simulations."""
    
    def __init__(self, n_workers: int = None, chunk_size: int = 1000):
        """Initialize parallel what-if simulator.
        
        Args:
            n_workers: Number of parallel workers (None = auto-detect)
            chunk_size: Size of processing chunks for batch operations
        """
        self.n_workers = n_workers or min(mp.cpu_count(), 4)
        self.chunk_size = chunk_size
        self.strategies = {
            "Parallel Random Search": ParallelRandomSearchStrategy
        }
        
    def simulate_alternative_strategy_parallel(self,
                                             optimizer,
                                             strategy_name: str,
                                             n_evaluations: int,
                                             param_bounds: Dict[str, Tuple[float, float]],
                                             seed: Optional[int] = None,
                                             progress_callback: Optional[callable] = None) -> Dict[str, Any]:
        """Simulate how an alternative strategy would have performed using parallel processing.
        
        Args:
            optimizer: Trained optimizer with final GP models
            strategy_name: Name of the strategy to simulate
            n_evaluations: Number of evaluations to simulate
            param_bounds: Parameter bounds for generating candidates
            seed: Random seed for reproducibility
            progress_callback: Optional callback for progress updates
            
        Returns:
            Dictionary containing simulation results with performance metrics
        """
        logger.info(f"Running parallel simulation: {strategy_name} with {n_evaluations} evaluations "
                   f"using {self.n_workers} workers")
        
        if strategy_name not in self.strategies:
            available = ", ".join(self.strategies.keys())
            raise ValueError(f"Unknown strategy '{strategy_name}'. Available: {available}")
        
        start_time = time.time()
        
        # Step 1: Parallel candidate generation
        if progress_callback:
            progress_callback("Generating candidates in parallel...")
            
        strategy = self.strategies[strategy_name](seed=seed)
        candidates_df = strategy.generate_candidates_batch(
            param_bounds=param_bounds,
            n_candidates=n_evaluations,
            batch_size=self.chunk_size,
            n_workers=self.n_workers,
            seed=seed
        )
        
        # Step 2: Parallel GP model predictions
        if progress_callback:
            progress_callback("Running batch predictions...")
            
        predictions_df = self._predict_with_optimizer_parallel(
            optimizer, candidates_df, progress_callback
        )
        
        # Step 3: Parallel hypervolume calculations
        if progress_callback:
            progress_callback("Computing hypervolume progression...")
            
        hypervolume_progression = self._calculate_hypervolume_progression_parallel(
            predictions_df, optimizer, progress_callback
        )
        
        total_time = time.time() - start_time
        throughput = n_evaluations / total_time
        
        logger.info(f"Parallel simulation completed in {total_time:.2f}s "
                   f"({throughput:.0f} evaluations/s)")
        
        return {
            'strategy_name': strategy_name,
            'candidates': candidates_df,
            'predictions': predictions_df,
            'hypervolume_progression': hypervolume_progression,
            'n_evaluations': n_evaluations,
            'execution_time': total_time,
            'throughput': throughput,
            'n_workers_used': self.n_workers,
            'chunk_size': self.chunk_size
        }
    
    def _predict_with_optimizer_parallel(self, optimizer, candidates_df: pd.DataFrame,
                                        progress_callback: Optional[callable] = None) -> pd.DataFrame:
        """Use the optimizer's GP models to predict responses in parallel batches.
        
        Args:
            optimizer: Trained optimizer (PyMBO EnhancedMultiObjectiveOptimizer)
            candidates_df: Candidate parameter sets
            progress_callback: Optional progress callback
            
        Returns:
            DataFrame with predicted responses
        """
        try:
            # Get parameter and response names
            param_names = list(candidates_df.columns)
            
            if hasattr(optimizer, 'responses_config') and optimizer.responses_config:
                response_names = list(optimizer.responses_config.keys())
            else:
                response_names = ['f1', 'f2']  # Default fallback
            
            logger.info(f"Using PyMBO optimizer with responses: {response_names}")
            
            # For large datasets, use chunked processing
            if len(candidates_df) > 1000:
                return self._predict_chunked_parallel(optimizer, candidates_df, response_names, progress_callback)
            else:
                return self._predict_sequential_with_optimizer(optimizer, candidates_df, response_names)
            
        except Exception as e:
            logger.warning(f"Optimizer prediction failed: {e}, falling back to random predictions")
            return self._generate_fallback_predictions(candidates_df, optimizer)
    
    def _predict_sequential_with_optimizer(self, optimizer, candidates_df: pd.DataFrame, 
                                         response_names: List[str]) -> pd.DataFrame:
        """Use optimizer to predict responses - now with batch optimization."""
        try:
            # Check if optimizer has batch prediction capability
            if hasattr(optimizer, 'predict_responses_batch'):
                logger.info("Using optimized batch prediction")
                return self._predict_with_batch_method(optimizer, candidates_df, response_names)
            else:
                logger.info("Enhancing optimizer with batch prediction capabilities")
                return self._predict_with_enhanced_batch(optimizer, candidates_df, response_names)
                
        except Exception as e:
            logger.error(f"Batch prediction failed, falling back to individual: {e}")
            return self._predict_individual_fallback(optimizer, candidates_df, response_names)
    
    def _predict_with_batch_method(self, optimizer, candidates_df: pd.DataFrame, 
                                 response_names: List[str]) -> pd.DataFrame:
        """Use existing batch prediction method."""
        param_dicts = candidates_df.to_dict('records')
        
        # Use batch prediction
        predictions_df = optimizer.predict_responses_batch(
            param_dicts=param_dicts,
            use_cache=True,
            batch_size=min(self.chunk_size, 200)  # Reasonable batch size
        )
        
        return predictions_df
    
    def _predict_with_enhanced_batch(self, optimizer, candidates_df: pd.DataFrame, 
                                   response_names: List[str]) -> pd.DataFrame:
        """Enhance optimizer with batch prediction and use it."""
        try:
            from pymbo.core.batch_prediction_optimizer import enhance_optimizer_with_batch_prediction
            
            # Enhance optimizer with batch capabilities
            enhanced_optimizer = enhance_optimizer_with_batch_prediction(optimizer)
            
            # Convert to parameter dictionaries
            param_dicts = candidates_df.to_dict('records') 
            
            # Use enhanced batch prediction
            predictions_df = enhanced_optimizer.predict_responses_batch(
                param_dicts=param_dicts,
                use_cache=True,
                batch_size=min(self.chunk_size, 200)
            )
            
            logger.info(f"Enhanced batch prediction completed for {len(predictions_df)} points")
            return predictions_df
            
        except Exception as e:
            logger.error(f"Enhanced batch prediction failed: {e}")
            return self._predict_individual_fallback(optimizer, candidates_df, response_names)
    
    def _predict_individual_fallback(self, optimizer, candidates_df: pd.DataFrame, 
                                   response_names: List[str]) -> pd.DataFrame:
        """Fallback to individual predictions (original slow method)."""
        logger.warning("Using slow individual predictions - consider upgrading optimizer")
        
        predictions = candidates_df.copy()
        
        # Use optimizer's predict_responses_at method if available
        for response_name in response_names:
            response_predictions = []
            
            for _, row in candidates_df.iterrows():
                param_dict = row.to_dict()
                
                # Try to use optimizer's prediction method
                try:
                    if hasattr(optimizer, 'predict_responses_at'):
                        pred_result = optimizer.predict_responses_at(param_dict)
                        if response_name in pred_result:
                            mean_val = pred_result[response_name].get('mean', 0.0)
                            response_predictions.append(float(mean_val))
                        else:
                            response_predictions.append(0.0)
                    else:
                        # Fallback to random prediction
                        response_predictions.append(np.random.normal(0, 1))
                except Exception as e:
                    logger.debug(f"Prediction failed for {response_name}: {e}")
                    response_predictions.append(np.random.normal(0, 1))
            
            predictions[response_name] = response_predictions
        
        return predictions
    
    def _predict_chunked_parallel(self, optimizer, candidates_df: pd.DataFrame, 
                                response_names: List[str], progress_callback: Optional[callable] = None) -> pd.DataFrame:
        """Use optimizer to predict responses in parallel chunks for large datasets."""
        try:
            # Split candidates into chunks
            n_chunks = (len(candidates_df) + self.chunk_size - 1) // self.chunk_size
            chunks = []
            
            for i in range(n_chunks):
                start_idx = i * self.chunk_size
                end_idx = min((i + 1) * self.chunk_size, len(candidates_df))
                chunk = candidates_df.iloc[start_idx:end_idx].copy()
                chunks.append(chunk)
            
            logger.info(f"Processing {len(candidates_df)} candidates in {n_chunks} chunks")
            
            # Process chunks sequentially due to GPU model serialization limitations
            # Note: BoTorch models with CUDA/MPS tensors cannot be pickled for multiprocessing
            all_predictions = []
            
            for i, chunk in enumerate(chunks):
                try:
                    chunk_predictions = self._predict_sequential_with_optimizer(optimizer, chunk, response_names)
                    all_predictions.append(chunk_predictions)
                    
                    if progress_callback:
                        progress = (i + 1) / n_chunks
                        progress_callback(f"Predictions: {i+1}/{n_chunks} chunks complete")
                        
                except Exception as e:
                    logger.error(f"Chunk {i} prediction failed: {e}")
                    # Create fallback predictions
                    fallback = chunk.copy()
                    for resp_name in response_names:
                        fallback[resp_name] = np.random.normal(0, 1, len(chunk))
                    all_predictions.append(fallback)
            
            # Concatenate all predictions
            predictions_df = pd.concat(all_predictions, ignore_index=True)
            return predictions_df
            
        except Exception as e:
            logger.error(f"Chunked prediction failed: {e}")
            return self._generate_fallback_predictions(candidates_df, optimizer)
    
    def _calculate_hypervolume_progression_parallel(self, predictions_df: pd.DataFrame, 
                                                   optimizer,
                                                   progress_callback: Optional[callable] = None) -> List[float]:
        """Calculate hypervolume progression using parallel processing.
        
        Args:
            predictions_df: DataFrame with predictions
            optimizer: Optimizer instance for context
            progress_callback: Optional progress callback
            
        Returns:
            List of hypervolume values at each step
        """
        try:
            # Identify response columns
            response_cols = []
            param_cols = []
            
            for col in predictions_df.columns:
                if hasattr(optimizer, 'responses_config'):
                    if col in optimizer.responses_config:
                        response_cols.append(col)
                    elif col in getattr(optimizer, 'params_config', {}):
                        param_cols.append(col)
                else:
                    if col.startswith('f') or col.lower() in ['yield', 'efficiency', 'cost', 'time']:
                        response_cols.append(col)
                    else:
                        param_cols.append(col)
            
            if not response_cols:
                response_cols = [col for col in predictions_df.columns if col not in param_cols]
            
            if not response_cols:
                response_cols = predictions_df.columns[-2:].tolist()
            
            logger.debug(f"Using response columns for hypervolume: {response_cols}")
            
            # For large datasets, use chunked parallel processing
            n_points = len(predictions_df)
            if n_points > 5000:  # Use parallel processing for large datasets
                return self._calculate_hypervolume_chunked_parallel(
                    predictions_df, response_cols, progress_callback
                )
            else:
                # For smaller datasets, use sequential processing to avoid overhead
                return self._calculate_hypervolume_sequential(predictions_df, response_cols)
                
        except Exception as e:
            logger.warning(f"Hypervolume progression calculation failed: {e}")
            return list(range(1, len(predictions_df) + 1))
    
    def _calculate_hypervolume_chunked_parallel(self, predictions_df: pd.DataFrame, 
                                               response_cols: List[str],
                                               progress_callback: Optional[callable] = None) -> List[float]:
        """Calculate hypervolume using chunked parallel processing for large datasets."""
        
        # Split data into chunks
        chunk_size = min(self.chunk_size, 2000)  # Smaller chunks for HV calculation
        n_chunks = (len(predictions_df) + chunk_size - 1) // chunk_size
        
        logger.info(f"Computing hypervolume for {len(predictions_df)} points in {n_chunks} chunks")
        
        all_hv_results = []
        
        with ThreadPoolExecutor(max_workers=self.n_workers) as executor:
            chunk_args = []
            for i in range(n_chunks):
                start_idx = i * chunk_size
                end_idx = min((i + 1) * chunk_size, len(predictions_df))
                chunk = predictions_df.iloc[start_idx:end_idx]
                chunk_args.append((chunk, response_cols, start_idx))
            
            future_to_chunk = {executor.submit(_calculate_hypervolume_chunk, args): i 
                             for i, args in enumerate(chunk_args)}
            
            completed_chunks = 0
            for future in as_completed(future_to_chunk):
                chunk_id = future_to_chunk[future]
                try:
                    start_idx, chunk_hv = future.result()
                    all_hv_results.append((start_idx, chunk_hv))
                    completed_chunks += 1
                    
                    if progress_callback:
                        progress = completed_chunks / n_chunks
                        progress_callback(f"Hypervolume: {completed_chunks}/{n_chunks} chunks complete")
                        
                except Exception as e:
                    logger.error(f"Hypervolume chunk {chunk_id} failed: {e}")
                    # Generate fallback values
                    start_idx = chunk_id * chunk_size
                    chunk_size_actual = min(chunk_size, len(predictions_df) - start_idx)
                    fallback_hv = [float(start_idx + j + 1) for j in range(chunk_size_actual)]
                    all_hv_results.append((start_idx, fallback_hv))
        
        # Sort results by start index and concatenate
        all_hv_results.sort(key=lambda x: x[0])
        progression = []
        for _, chunk_hv in all_hv_results:
            progression.extend(chunk_hv)
        
        return progression
    
    def _calculate_hypervolume_sequential(self, predictions_df: pd.DataFrame, 
                                         response_cols: List[str]) -> List[float]:
        """Calculate hypervolume sequentially for smaller datasets."""
        progression = []
        Y_cumulative = []
        
        for i, (_, row) in enumerate(predictions_df.iterrows()):
            y_current = row[response_cols].values
            y_current = np.array(y_current, dtype=float)
            
            if not np.all(np.isfinite(y_current)):
                y_current = np.nan_to_num(y_current, nan=0.0, posinf=0.0, neginf=0.0)
            
            Y_cumulative.append(y_current)
            Y_array = np.array(Y_cumulative, dtype=float)
            
            # Simple hypervolume approximation
            if Y_array.shape[0] > 0 and Y_array.shape[1] > 0:
                ranges = np.ptp(Y_array, axis=0)
                hv_approx = np.prod(ranges) * Y_array.shape[0]
            else:
                hv_approx = 0.0
                
            progression.append(float(hv_approx))
        
        return progression
    
    def _generate_fallback_predictions(self, candidates_df: pd.DataFrame, optimizer) -> pd.DataFrame:
        """Generate fallback predictions when GP models are not available."""
        logger.warning("Using fallback random predictions")
        
        predictions = candidates_df.copy()
        
        try:
            if hasattr(optimizer, 'responses_config'):
                response_names = list(optimizer.responses_config.keys())
            elif hasattr(optimizer, '_responses_config'):
                response_names = list(optimizer._responses_config.keys())
            else:
                response_names = ['f1', 'f2']
                
            for response_name in response_names:
                predictions[response_name] = np.random.normal(0, 1, len(candidates_df))
                
        except Exception as e:
            logger.warning(f"Could not determine responses: {e}")
            predictions['f1'] = np.random.normal(0, 1, len(candidates_df))
            predictions['f2'] = np.random.normal(0, 1, len(candidates_df))
            
        return predictions
    
    def get_parameter_bounds_from_optimizer(self, optimizer) -> Dict[str, Tuple[float, float]]:
        """Extract parameter bounds from the optimizer."""
        bounds = {}
        
        try:
            if hasattr(optimizer, 'params_config'):
                params_config = optimizer.params_config
            elif hasattr(optimizer, '_params_config'):
                params_config = optimizer._params_config
            else:
                raise AttributeError("No parameter configuration found")
                
            for param_name, config in params_config.items():
                if config.get('type') == 'continuous' and 'bounds' in config:
                    bounds[param_name] = tuple(config['bounds'])
                else:
                    bounds[param_name] = (0.0, 1.0)
                    
        except Exception as e:
            logger.warning(f"Could not extract parameter bounds: {e}")
            bounds = {'x1': (0.0, 1.0), 'x2': (0.0, 1.0)}
            
        return bounds


def create_parallel_whatif_report(simulation_results: Dict[str, Any]) -> str:
    """Create a detailed text report from parallel what-if simulation results.
    
    Args:
        simulation_results: Results from ParallelWhatIfSimulator.simulate_alternative_strategy_parallel()
        
    Returns:
        Formatted text report with performance metrics
    """
    report = []
    report.append("=" * 60)
    report.append("PARALLEL WHAT-IF SIMULATION REPORT")
    report.append("=" * 60)
    report.append("")
    
    report.append(f"Strategy: {simulation_results['strategy_name']}")
    report.append(f"Simulated Evaluations: {simulation_results['n_evaluations']}")
    report.append("")
    
    # Performance metrics
    report.append("Performance Metrics:")
    report.append(f"  Execution Time: {simulation_results['execution_time']:.2f} seconds")
    report.append(f"  Throughput: {simulation_results['throughput']:.0f} evaluations/second")
    report.append(f"  Workers Used: {simulation_results['n_workers_used']}")
    report.append(f"  Chunk Size: {simulation_results['chunk_size']}")
    report.append("")
    
    hv_progression = simulation_results['hypervolume_progression']
    if hv_progression:
        final_hv = hv_progression[-1]
        report.append(f"Final Hypervolume: {final_hv:.6f}")
        
        if len(hv_progression) > 1:
            improvement = hv_progression[-1] - hv_progression[0]
            report.append(f"Total Improvement: {improvement:.6f}")
            
            # Calculate convergence statistics
            mid_point = len(hv_progression) // 2
            early_avg = np.mean(hv_progression[:mid_point])
            late_avg = np.mean(hv_progression[mid_point:])
            
            report.append(f"Early Phase Avg HV: {early_avg:.6f}")
            report.append(f"Late Phase Avg HV: {late_avg:.6f}")
            report.append(f"Phase Improvement: {late_avg - early_avg:.6f}")
            
    report.append("")
    report.append("This parallel simulation shows how the alternative strategy")
    report.append("would have performed using the same evaluation budget")
    report.append("and the final trained surrogate models, with significant")
    report.append("performance improvements through parallel processing.")
    report.append("")
    report.append("=" * 60)
    
    return "\n".join(report)