"""
Plot Data Fixer for PyMBO Validation Results
===========================================

This module fixes corrupted plot data by ensuring consistent history storage
and proper hypervolume progression calculation. It addresses the issues causing
incomplete plots and missing data points in validation results.

Key fixes:
- Consistent history intervals for all algorithms
- Complete hypervolume progression from evaluation 1 to n
- Proper data synchronization between algorithms
- Robust error handling for missing data
- Fixed convergence rate calculations

Author: Multi-Objective Optimization Laboratory
Version: 1.0.0 - Plot Data Fix
"""

import logging
import numpy as np
import pandas as pd
from typing import Any, Dict, List, Optional, Tuple

logger = logging.getLogger(__name__)


class ValidationDataFixer:
    """Fixes corrupted validation data for proper plotting."""
    
    def __init__(self):
        """Initialize the data fixer."""
        self.fixed_data = {}
    
    def fix_validation_results(self, validation_results: Dict[str, Any]) -> Dict[str, Any]:
        """
        Fix corrupted validation results to ensure proper plotting.
        
        Args:
            validation_results: Original validation results
            
        Returns:
            Fixed validation results with complete data
        """
        logger.info("🔧 Fixing corrupted validation data for proper plotting...")
        
        fixed_results = validation_results.copy()
        
        # Fix hypervolume progression data
        if 'hypervolume_results' in fixed_results:
            fixed_results['hypervolume_results'] = self._fix_hypervolume_progression(
                fixed_results['hypervolume_results']
            )
        
        # Fix benchmark results history
        if 'benchmark_results' in fixed_results:
            fixed_results['benchmark_results'] = self._fix_benchmark_history(
                fixed_results['benchmark_results'], 
                fixed_results.get('n_evaluations', 15)
            )
        
        # Create proper hypervolume progression structure for GUI
        fixed_results['hypervolume_progression'] = self._create_hypervolume_progression_structure(
            fixed_results
        )
        
        logger.info("✅ Validation data fixed successfully")
        return fixed_results
    
    def _fix_hypervolume_progression(self, hv_results: Dict[str, Any]) -> Dict[str, Any]:
        """Fix hypervolume progression data to ensure complete sequences."""
        fixed_hv = {}
        
        for alg_name, alg_data in hv_results.items():
            logger.debug(f"🔧 Fixing hypervolume data for {alg_name}")
            
            if not alg_data or 'mean' not in alg_data:
                # Create empty structure
                fixed_hv[alg_name] = {
                    'mean': [],
                    'std': [],
                    'all_runs': [],
                    'evaluations': []
                }
                continue
            
            mean_vals = np.array(alg_data['mean']) if len(alg_data['mean']) > 0 else np.array([])
            std_vals = np.array(alg_data['std']) if len(alg_data['std']) > 0 else np.zeros_like(mean_vals)
            
            if len(mean_vals) == 0:
                fixed_hv[alg_name] = {
                    'mean': [],
                    'std': [],
                    'all_runs': [],
                    'evaluations': []
                }
                continue
            
            # Ensure we have complete evaluation sequence
            max_evals = len(mean_vals)
            evaluations = list(range(1, max_evals + 1))
            
            # Fix any NaN or infinite values
            mean_vals = self._fix_invalid_values(mean_vals)
            std_vals = self._fix_invalid_values(std_vals)
            
            # Ensure std has same length as mean
            if len(std_vals) != len(mean_vals):
                std_vals = np.zeros_like(mean_vals)
            
            fixed_hv[alg_name] = {
                'mean': mean_vals.tolist(),
                'std': std_vals.tolist(), 
                'all_runs': alg_data.get('all_runs', []),
                'evaluations': evaluations
            }
            
            logger.debug(f"  Fixed {alg_name}: {len(mean_vals)} evaluations")
        
        return fixed_hv
    
    def _fix_benchmark_history(self, benchmark_results: Dict[str, List], n_evaluations: int) -> Dict[str, List]:
        """Fix benchmark history to ensure consistent data intervals."""
        fixed_benchmark = {}
        
        for alg_name, runs in benchmark_results.items():
            logger.debug(f"🔧 Fixing benchmark history for {alg_name}")
            
            fixed_runs = []
            
            for run_idx, run_data in enumerate(runs):
                if 'history' not in run_data:
                    # Create minimal history from X, Y data
                    run_data['history'] = self._create_history_from_xy(
                        run_data.get('X', np.array([])), 
                        run_data.get('Y', np.array([]))
                    )
                
                # Fix history to have consistent intervals
                fixed_history = self._fix_history_intervals(
                    run_data['history'], n_evaluations
                )
                
                fixed_run = run_data.copy()
                fixed_run['history'] = fixed_history
                fixed_runs.append(fixed_run)
            
            fixed_benchmark[alg_name] = fixed_runs
            logger.debug(f"  Fixed {alg_name}: {len(fixed_runs)} runs")
        
        return fixed_benchmark
    
    def _create_history_from_xy(self, X: np.ndarray, Y: np.ndarray) -> List[Dict]:
        """Create history structure from X, Y arrays."""
        history = []
        
        if len(X) == 0 or len(Y) == 0:
            return history
        
        # Create history entries for key evaluation points
        total_evals = len(X)
        
        # Store history at regular intervals
        interval = max(1, total_evals // 10)  # At least 10 points
        
        for i in range(interval, total_evals + 1, interval):
            if i <= total_evals:
                history.append({
                    'X': X[:i].copy(),
                    'Y': Y[:i].copy(),
                    'evaluation': i
                })
        
        # Always include the final evaluation
        if total_evals not in [h['evaluation'] for h in history]:
            history.append({
                'X': X.copy(),
                'Y': Y.copy(), 
                'evaluation': total_evals
            })
        
        return history
    
    def _fix_history_intervals(self, history: List[Dict], n_evaluations: int) -> List[Dict]:
        """Fix history to have consistent evaluation intervals."""
        if not history:
            return []
        
        # Sort history by evaluation number
        sorted_history = sorted(history, key=lambda h: h.get('evaluation', 0))
        
        fixed_history = []
        
        # Ensure we have entries at regular intervals
        target_intervals = max(1, n_evaluations // 10)  # ~10 data points
        
        for i in range(target_intervals, n_evaluations + 1, target_intervals):
            # Find closest history entry
            closest_entry = None
            min_diff = float('inf')
            
            for entry in sorted_history:
                eval_num = entry.get('evaluation', 0)
                diff = abs(eval_num - i)
                if diff < min_diff:
                    min_diff = diff
                    closest_entry = entry
            
            if closest_entry:
                fixed_entry = closest_entry.copy()
                fixed_entry['evaluation'] = i  # Normalize evaluation number
                fixed_history.append(fixed_entry)
        
        # Always include final evaluation
        if sorted_history and sorted_history[-1]['evaluation'] != n_evaluations:
            final_entry = sorted_history[-1].copy()
            final_entry['evaluation'] = n_evaluations
            fixed_history.append(final_entry)
        
        return fixed_history
    
    def _create_hypervolume_progression_structure(self, validation_results: Dict[str, Any]) -> Dict[str, Dict]:
        """Create proper hypervolume progression structure for GUI plotting."""
        test_function = validation_results.get('test_function', 'Unknown')
        hv_results = validation_results.get('hypervolume_results', {})
        
        progression_structure = {
            test_function: {}
        }
        
        for alg_name, alg_data in hv_results.items():
            if alg_data and 'mean' in alg_data and alg_data['mean']:
                progression_structure[test_function][alg_name] = {
                    'mean': alg_data['mean'],
                    'std': alg_data.get('std', [0] * len(alg_data['mean'])),
                    'evaluations': alg_data.get('evaluations', list(range(1, len(alg_data['mean']) + 1)))
                }
        
        return progression_structure
    
    def _fix_invalid_values(self, values: np.ndarray) -> np.ndarray:
        """Fix NaN, infinite, or invalid values in arrays."""
        if len(values) == 0:
            return values
        
        # Replace NaN and infinite values
        values = np.array(values, dtype=float)
        mask = np.isfinite(values)
        
        if not np.any(mask):
            # All values are invalid, create zeros
            return np.zeros_like(values)
        
        # Forward fill NaN values
        if not np.all(mask):
            # Get first valid value
            first_valid_idx = np.where(mask)[0][0]
            first_valid_val = values[first_valid_idx]
            
            # Fill initial NaN values
            values[:first_valid_idx] = first_valid_val
            
            # Forward fill remaining NaN values
            for i in range(1, len(values)):
                if not np.isfinite(values[i]):
                    values[i] = values[i-1]
        
        return values
    
    def validate_fixed_data(self, fixed_results: Dict[str, Any]) -> bool:
        """Validate that the fixed data is suitable for plotting."""
        try:
            hv_prog = fixed_results.get('hypervolume_progression', {})
            
            for test_func, alg_data in hv_prog.items():
                for alg_name, data in alg_data.items():
                    mean_vals = data.get('mean', [])
                    evals = data.get('evaluations', [])
                    
                    if len(mean_vals) == 0:
                        logger.warning(f"No data for {alg_name} in {test_func}")
                        continue
                    
                    if len(mean_vals) != len(evals):
                        logger.error(f"Length mismatch for {alg_name}: mean={len(mean_vals)}, evals={len(evals)}")
                        return False
                    
                    # Check for valid values
                    if not all(np.isfinite(mean_vals)):
                        logger.error(f"Invalid values in {alg_name} data")
                        return False
            
            logger.info("✅ Fixed data validation passed")
            return True
            
        except Exception as e:
            logger.error(f"Data validation failed: {e}")
            return False


def fix_validation_results_for_plotting(validation_results: Dict[str, Any]) -> Dict[str, Any]:
    """
    Convenience function to fix validation results for proper plotting.
    
    Args:
        validation_results: Original validation results
        
    Returns:
        Fixed validation results
    """
    fixer = ValidationDataFixer()
    fixed_results = fixer.fix_validation_results(validation_results)
    
    if fixer.validate_fixed_data(fixed_results):
        return fixed_results
    else:
        logger.error("Data fixing failed validation, returning original results")
        return validation_results


# Enhanced algorithms that store complete history
class FixedHistoryMixin:
    """Mixin to ensure algorithms store complete history for plotting."""
    
    def _store_complete_history(self, X_all: np.ndarray, Y_all: np.ndarray, n_evaluations: int) -> List[Dict]:
        """Store complete history at regular intervals for consistent plotting."""
        history = []
        
        if len(X_all) == 0:
            return history
        
        # Store history every few evaluations for smooth plots
        interval = max(1, n_evaluations // 20)  # ~20 data points minimum
        
        for i in range(interval, len(X_all) + 1, interval):
            history.append({
                'X': X_all[:i].copy(),
                'Y': Y_all[:i].copy(),
                'evaluation': i
            })
        
        # Always store final evaluation
        if len(X_all) not in [h['evaluation'] for h in history]:
            history.append({
                'X': X_all.copy(),
                'Y': Y_all.copy(),
                'evaluation': len(X_all)
            })
        
        return history