"""
What-If Simulation Module - Post-Hoc Analysis of Alternative Strategies

This module provides functionality to simulate how alternative optimization strategies
would have performed using the same evaluation budget as a completed experiment.
It uses the final trained Gaussian Process models as a "virtual laboratory" to
predict the performance of naive strategies like Random Search.

Classes:
    WhatIfSimulator: Main class for running what-if simulations
    SimulationStrategy: Base class for simulation strategies
    RandomSearchStrategy: Random search simulation implementation

Author: Multi-Objective Optimization Laboratory
Version: 3.2.0 Enhanced
"""

import numpy as np
import pandas as pd
from typing import Dict, List, Optional, Tuple, Any, Union
import logging
import warnings
from abc import ABC, abstractmethod
import torch

# Suppress warnings for cleaner output
warnings.filterwarnings("ignore", category=UserWarning)

logger = logging.getLogger(__name__)


class SimulationStrategy(ABC):
    """Base class for what-if simulation strategies."""
    
    def __init__(self, name: str):
        """Initialize simulation strategy.
        
        Args:
            name: Strategy name for identification
        """
        self.name = name
        
    @abstractmethod
    def generate_candidates(self, 
                          param_bounds: Dict[str, Tuple[float, float]], 
                          n_candidates: int,
                          seed: Optional[int] = None) -> pd.DataFrame:
        """Generate candidate parameter sets for the strategy.
        
        Args:
            param_bounds: Parameter bounds dictionary
            n_candidates: Number of candidates to generate
            seed: Random seed for reproducibility
            
        Returns:
            DataFrame with candidate parameter sets
        """
        pass


class RandomSearchStrategy(SimulationStrategy):
    """Random search simulation strategy."""
    
    def __init__(self, seed: Optional[int] = None):
        """Initialize random search strategy.
        
        Args:
            seed: Random seed for reproducibility
        """
        super().__init__("Random Search")
        self.rng = np.random.RandomState(seed)
        
    def generate_candidates(self, 
                          param_bounds: Dict[str, Tuple[float, float]], 
                          n_candidates: int,
                          seed: Optional[int] = None) -> pd.DataFrame:
        """Generate random candidate parameter sets.
        
        Args:
            param_bounds: Parameter bounds dictionary
            n_candidates: Number of candidates to generate
            seed: Random seed (overrides instance seed if provided)
            
        Returns:
            DataFrame with random parameter sets
        """
        if seed is not None:
            rng = np.random.RandomState(seed)
        else:
            rng = self.rng
            
        candidates = {}
        
        for param_name, (lower, upper) in param_bounds.items():
            candidates[param_name] = rng.uniform(
                low=lower, 
                high=upper, 
                size=n_candidates
            )
            
        return pd.DataFrame(candidates)


class WhatIfSimulator:
    """Main class for running what-if simulations."""
    
    def __init__(self):
        """Initialize what-if simulator."""
        self.strategies = {
            "Random Search": RandomSearchStrategy
        }
        
    def simulate_alternative_strategy(self,
                                    optimizer,
                                    strategy_name: str,
                                    n_evaluations: int,
                                    param_bounds: Dict[str, Tuple[float, float]],
                                    seed: Optional[int] = None) -> Dict[str, Any]:
        """Simulate how an alternative strategy would have performed.
        
        Args:
            optimizer: Trained optimizer with final GP models
            strategy_name: Name of the strategy to simulate
            n_evaluations: Number of evaluations to simulate
            param_bounds: Parameter bounds for generating candidates
            seed: Random seed for reproducibility
            
        Returns:
            Dictionary containing simulation results
        """
        logger.info(f"Simulating {strategy_name} with {n_evaluations} evaluations")
        
        if strategy_name not in self.strategies:
            available = ", ".join(self.strategies.keys())
            raise ValueError(f"Unknown strategy '{strategy_name}'. Available: {available}")
            
        # Create strategy instance
        strategy = self.strategies[strategy_name](seed=seed)
        
        # Generate candidate parameter sets
        candidates_df = strategy.generate_candidates(
            param_bounds=param_bounds,
            n_candidates=n_evaluations,
            seed=seed
        )
        
        # Use the final GP models to predict responses
        try:
            # Try to get predictions from the optimizer
            predictions_df = self._predict_with_optimizer(optimizer, candidates_df)
        except Exception as e:
            logger.warning(f"Could not use optimizer for predictions: {e}")
            # Fallback to random predictions
            predictions_df = self._generate_fallback_predictions(candidates_df, optimizer)
            
        # Calculate hypervolume progression
        hypervolume_progression = self._calculate_hypervolume_progression(
            predictions_df, optimizer
        )
        
        return {
            'strategy_name': strategy_name,
            'candidates': candidates_df,
            'predictions': predictions_df,
            'hypervolume_progression': hypervolume_progression,
            'n_evaluations': n_evaluations
        }
        
    def _predict_with_optimizer(self, optimizer, candidates_df: pd.DataFrame) -> pd.DataFrame:
        """Use the optimizer's GP models to predict responses.
        
        Args:
            optimizer: Trained optimizer
            candidates_df: Candidate parameter sets
            
        Returns:
            DataFrame with predicted responses
        """
        # This method tries to use the actual optimizer's GP models
        # The exact implementation depends on the optimizer's interface
        
        try:
            # Check if optimizer has the method to predict
            if hasattr(optimizer, 'predict_responses_at'):
                # Use predict_responses_at method for each candidate
                predictions = candidates_df.copy()
                
                # Get response names
                if hasattr(optimizer, 'responses_config'):
                    response_names = list(optimizer.responses_config.keys())
                else:
                    response_names = ['f1', 'f2']  # Default fallback
                
                # Predict each response for each candidate
                for response_name in response_names:
                    response_predictions = []
                    for _, row in candidates_df.iterrows():
                        param_dict = row.to_dict()
                        pred_result = optimizer.predict_responses_at(param_dict)
                        if response_name in pred_result:
                            response_predictions.append(pred_result[response_name].get('mean', 0.0))
                        else:
                            response_predictions.append(0.0)
                    predictions[response_name] = response_predictions
                
                return predictions
            elif hasattr(optimizer, 'predict_responses'):
                predictions = optimizer.predict_responses(candidates_df)
                return predictions
            elif hasattr(optimizer, '_models') and optimizer._models:
                # Try to use internal models directly
                return self._predict_with_models(optimizer._models, candidates_df, optimizer)
            else:
                raise AttributeError("No prediction method available")
                
        except Exception as e:
            logger.warning(f"Prediction with optimizer failed: {e}")
            raise
            
    def _predict_with_models(self, models, candidates_df: pd.DataFrame, optimizer) -> pd.DataFrame:
        """Use GP models directly for prediction.
        
        Args:
            models: Dictionary of GP models
            candidates_df: Candidate parameter sets
            optimizer: Optimizer instance for context
            
        Returns:
            DataFrame with predicted responses
        """
        import torch
        
        # Convert candidates to tensor format expected by models
        param_names = list(candidates_df.columns)
        X_tensor = torch.tensor(candidates_df.values, dtype=torch.float32)
        
        predictions = {}
        
        # Add parameter columns to predictions
        for param in param_names:
            predictions[param] = candidates_df[param].values
            
        # Predict each response
        for response_name, model in models.items():
            try:
                with torch.no_grad():
                    posterior = model.posterior(X_tensor)
                    pred_mean = posterior.mean.numpy().flatten()
                    predictions[response_name] = pred_mean
                    
            except Exception as e:
                logger.warning(f"Could not predict {response_name}: {e}")
                # Generate random predictions as fallback
                predictions[response_name] = np.random.normal(0, 1, len(candidates_df))
                
        return pd.DataFrame(predictions)
        
    def _generate_fallback_predictions(self, candidates_df: pd.DataFrame, optimizer) -> pd.DataFrame:
        """Generate fallback predictions when GP models are not available.
        
        Args:
            candidates_df: Candidate parameter sets
            optimizer: Optimizer instance for context
            
        Returns:
            DataFrame with fallback predictions
        """
        logger.warning("Using fallback random predictions")
        
        predictions = candidates_df.copy()
        
        # Try to get response names from optimizer
        try:
            if hasattr(optimizer, 'responses_config'):
                response_names = list(optimizer.responses_config.keys())
            elif hasattr(optimizer, '_responses_config'):
                response_names = list(optimizer._responses_config.keys())
            else:
                response_names = ['f1', 'f2']  # Default fallback
                
            # Generate random predictions for each response
            for response_name in response_names:
                predictions[response_name] = np.random.normal(0, 1, len(candidates_df))
                
        except Exception as e:
            logger.warning(f"Could not determine responses: {e}")
            # Ultimate fallback
            predictions['f1'] = np.random.normal(0, 1, len(candidates_df))
            predictions['f2'] = np.random.normal(0, 1, len(candidates_df))
            
        return predictions
        
    def _calculate_hypervolume_progression(self, predictions_df: pd.DataFrame, optimizer) -> List[float]:
        """Calculate hypervolume progression for the simulated strategy.
        
        Args:
            predictions_df: DataFrame with predictions
            optimizer: Optimizer instance for context
            
        Returns:
            List of hypervolume values at each step
        """
        try:
            # Try to use existing hypervolume calculator
            from pymbo.core.hypervolume_calculator import HypervolumeCalculator
            hv_calc = HypervolumeCalculator()
            
            # Get response columns
            response_cols = []
            param_cols = []
            
            for col in predictions_df.columns:
                if hasattr(optimizer, 'responses_config'):
                    if col in optimizer.responses_config:
                        response_cols.append(col)
                    elif col in getattr(optimizer, 'params_config', {}):
                        param_cols.append(col)
                else:
                    # Heuristic: assume non-parameter columns are responses
                    if col.startswith('f') or col.lower() in ['yield', 'efficiency', 'cost', 'time']:
                        response_cols.append(col)
                    else:
                        param_cols.append(col)
                        
            if not response_cols:
                response_cols = [col for col in predictions_df.columns if col not in param_cols]
                
            if not response_cols:
                # Ultimate fallback
                response_cols = predictions_df.columns[-2:].tolist()
                
            logger.debug(f"Using response columns: {response_cols}")
            
            # Calculate progressive hypervolume
            progression = []
            Y_cumulative = []
            
            for i in range(len(predictions_df)):
                # Add current point
                y_current = predictions_df[response_cols].iloc[i].values
                
                # Ensure y_current is properly formatted and finite
                y_current = np.array(y_current, dtype=float)
                if not np.all(np.isfinite(y_current)):
                    logger.warning(f"Non-finite values in prediction at step {i}, replacing with zeros")
                    y_current = np.nan_to_num(y_current, nan=0.0, posinf=0.0, neginf=0.0)
                
                Y_cumulative.append(y_current)
                
                # Calculate hypervolume for accumulated points
                Y_array = np.array(Y_cumulative, dtype=float)
                
                # Ensure Y_array has correct shape (n_points, n_objectives)
                if Y_array.ndim == 1:
                    Y_array = Y_array.reshape(1, -1)
                elif Y_array.shape[0] == 0:
                    logger.warning(f"Empty Y_array at step {i}")
                    progression.append(0.0)
                    continue
                
                try:
                    # Convert to torch tensor with proper device handling
                    Y_tensor = torch.tensor(Y_array, dtype=torch.float32)
                    
                    # Move to same device as hypervolume calculator if needed
                    if hasattr(hv_calc, 'device'):
                        Y_tensor = Y_tensor.to(hv_calc.device)
                    
                    # Get objective names matching the response columns
                    objective_names = response_cols[:Y_array.shape[1]]
                    
                    # Ensure we have enough objective names
                    while len(objective_names) < Y_array.shape[1]:
                        objective_names.append(f"f{len(objective_names)+1}")
                    
                    logger.debug(f"Step {i}: Y_tensor shape: {Y_tensor.shape}, objectives: {objective_names}")
                    
                    hv_result = hv_calc.calculate_hypervolume(Y_tensor, objective_names)
                    hv = hv_result.get('hypervolume', 0.0) if isinstance(hv_result, dict) else float(hv_result)
                    
                    # Ensure hypervolume is finite
                    if not np.isfinite(hv):
                        hv = 0.0
                        
                    progression.append(hv)
                    logger.debug(f"Step {i}: Hypervolume = {hv}")
                    
                except Exception as e:
                    logger.warning(f"Hypervolume calculation failed at step {i}: {e}")
                    logger.debug(f"Y_array shape: {Y_array.shape}, dtype: {Y_array.dtype}")
                    # Use simple approximation based on number of non-dominated points
                    try:
                        # Simple approximation: use the volume of the bounding box
                        if Y_array.shape[0] > 0 and Y_array.shape[1] > 0:
                            ranges = np.ptp(Y_array, axis=0)  # Range for each objective
                            hv_approx = np.prod(ranges) * Y_array.shape[0]
                        else:
                            hv_approx = 0.0
                        progression.append(float(hv_approx))
                    except:
                        progression.append(float(i + 1))  # Last resort: just use step number
                    
        except Exception as e:
            logger.warning(f"Hypervolume progression calculation failed: {e}")
            # Fallback to simple progression
            progression = list(range(1, len(predictions_df) + 1))
            
        return progression
        
    def get_parameter_bounds_from_optimizer(self, optimizer) -> Dict[str, Tuple[float, float]]:
        """Extract parameter bounds from the optimizer.
        
        Args:
            optimizer: Optimizer instance
            
        Returns:
            Dictionary mapping parameter names to (min, max) bounds
        """
        bounds = {}
        
        try:
            if hasattr(optimizer, 'params_config'):
                params_config = optimizer.params_config
            elif hasattr(optimizer, '_params_config'):
                params_config = optimizer._params_config
            else:
                raise AttributeError("No parameter configuration found")
                
            for param_name, config in params_config.items():
                if config.get('type') == 'continuous' and 'bounds' in config:
                    bounds[param_name] = tuple(config['bounds'])
                else:
                    # Default bounds
                    bounds[param_name] = (0.0, 1.0)
                    
        except Exception as e:
            logger.warning(f"Could not extract parameter bounds: {e}")
            # Fallback bounds
            bounds = {'x1': (0.0, 1.0), 'x2': (0.0, 1.0)}
            
        return bounds


def create_whatif_report(simulation_results: Dict[str, Any]) -> str:
    """Create a text report from what-if simulation results.
    
    Args:
        simulation_results: Results from WhatIfSimulator.simulate_alternative_strategy()
        
    Returns:
        Formatted text report
    """
    report = []
    report.append("=" * 50)
    report.append("WHAT-IF SIMULATION REPORT")
    report.append("=" * 50)
    report.append("")
    
    report.append(f"Strategy: {simulation_results['strategy_name']}")
    report.append(f"Simulated Evaluations: {simulation_results['n_evaluations']}")
    report.append("")
    
    hv_progression = simulation_results['hypervolume_progression']
    if hv_progression:
        final_hv = hv_progression[-1]
        report.append(f"Final Hypervolume: {final_hv:.6f}")
        
        # Calculate some basic statistics
        if len(hv_progression) > 1:
            improvement = hv_progression[-1] - hv_progression[0]
            report.append(f"Total Improvement: {improvement:.6f}")
            
    report.append("")
    report.append("This simulation shows how the alternative strategy")
    report.append("would have performed using the same evaluation budget")
    report.append("and the final trained surrogate models.")
    report.append("")
    report.append("=" * 50)
    
    return "\n".join(report)