"""Data models for WirePusher library."""

from dataclasses import asdict, dataclass
from typing import Any, Dict, List, Optional


@dataclass
class NotificationRequest:
    """Notification request model for v1 API.

    Note: Token is sent via Authorization header, not in request body.

    Args:
        title: Notification title
        message: Notification message body
        type: Optional notification type for filtering/organization
        tags: Optional list of tags
        imageURL: Optional image URL to display
        actionURL: Optional URL to open when tapped
        iv: Optional initialization vector for encrypted messages (hex string, 32 chars)
    """

    title: str
    message: str
    type: Optional[str] = None
    tags: Optional[List[str]] = None
    imageURL: Optional[str] = None
    actionURL: Optional[str] = None
    iv: Optional[str] = None

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary, excluding None values.

        Returns:
            Dictionary representation suitable for JSON serialization.
        """
        data = asdict(self)
        return {k: v for k, v in data.items() if v is not None}


@dataclass
class NotificationResponse:
    """Notification response model from WirePusher API.

    Args:
        status: Response status (typically "success" or "error")
        message: Human-readable response message
    """

    status: str
    message: str


@dataclass
class NotifAIRequest:
    """NotifAI request model for AI-powered notifications.

    Note: Token is sent via Authorization header, not in request body.

    Args:
        text: Free-form text to convert into structured notification
        type: Optional notification type to override AI-generated type
    """

    text: str
    type: Optional[str] = None

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary, excluding None values.

        Returns:
            Dictionary representation suitable for JSON serialization.
        """
        data = asdict(self)
        return {k: v for k, v in data.items() if v is not None}


@dataclass
class NotifAIResponse:
    """NotifAI response model from WirePusher API.

    Args:
        status: Response status (typically "success" or "error")
        message: Human-readable response message
        notification: Optional structured notification generated by AI
    """

    status: str
    message: str
    notification: Optional[Dict[str, Any]] = None
