from typing import List
from django.conf import settings
from django.contrib.auth.models import Permission, Group
from django.apps import apps as django_apps
from django.core.exceptions import ImproperlyConfigured
from django.utils import timezone


def set_profile(user, groups: List[Group], permissions: List[Permission], user_info=None):
    user.last_login = timezone.now()
    user.groups = groups
    user.permissions = permissions
    if user_info:
        user = fill_user_model_keycloak(user=user, user_info=user_info)
    user.save()
    return user


def fill_user_model_keycloak(user, user_info):
    """
    Return the User Keycloak model that is active in this project.
    """
    if hasattr(user, 'is_superuser'):
        user.is_superuser = user_info.get('is_superuser', False)
    if hasattr(user, 'is_staff'):
        user.is_staff = user_info.get('is_staff', True)
    if hasattr(user, 'realm_name'):
        user.realm_name = user_info.get('realm_name', 'master')
    return user


def get_user_model_keycloak():
    """
    Return the User Keycloak model that is active in this project.
    """
    try:
        return django_apps.get_model(settings.KEYCLOAK['KEYCLOAK_USER_MODEL'], require_ready=False)
    except ValueError:
        raise ImproperlyConfigured(
            "KEYCLOAK_USER_MODEL must be of the form 'app_label.model_name'")
    except LookupError:
        raise ImproperlyConfigured(
            "KEYCLOAK_USER_MODEL refers to model '%s' that has not been installed" % settings.KEYCLOAK[
                'KEYCLOAK_USER_MODEL']
        )
