import { IAspect, IConstruct, CfnResource } from '@aws-cdk/core';
import { NagRuleCompliance } from './nag-rules';
import { NagPackSuppression } from './nag-suppressions';
/**
 * Interface for creating a Nag rule pack.
 *
 * @stability stable
 */
export interface NagPackProps {
    /**
     * Whether or not to enable extended explanatory descriptions on warning, error, and logged ignore messages (default: false).
     *
     * @stability stable
     */
    readonly verbose?: boolean;
    /**
     * Whether or not to log triggered rules that have been suppressed as informational messages (default: false).
     *
     * @stability stable
     */
    readonly logIgnores?: boolean;
    /**
     * Whether or not to generate CSV compliance reports for applied Stacks in the App's output directory (default: true).
     *
     * @stability stable
     */
    readonly reports?: boolean;
}
/**
 * Interface for JSII interoperability for passing parameters and the Rule Callback to @applyRule method.
 *
 * @stability stable
 */
export interface IApplyRule {
    /**
     * Override for the suffix of the Rule ID for this rule.
     *
     * @stability stable
     */
    ruleSuffixOverride?: string;
    /**
     * Why the rule was triggered.
     *
     * @stability stable
     */
    info: string;
    /**
     * Why the rule exists.
     *
     * @stability stable
     */
    explanation: string;
    /**
     * The annotations message level to apply to the rule if triggered.
     *
     * @stability stable
     */
    level: NagMessageLevel;
    /**
     * Ignores listed in cdk-nag metadata.
     *
     * @stability stable
     */
    node: CfnResource;
    /**
     * The callback to the rule.
     *
     * @param node The CfnResource to check.
     * @stability stable
     */
    rule(node: CfnResource): NagRuleCompliance;
}
/**
 * The level of the message that the rule applies.
 *
 * @stability stable
 */
export declare enum NagMessageLevel {
    /**
     * @stability stable
     */
    WARN = "Warning",
    /**
     * @stability stable
     */
    ERROR = "Error"
}
/**
 * Base class for all rule packs.
 *
 * @stability stable
 */
export declare abstract class NagPack implements IAspect {
    /**
     * @stability stable
     */
    protected verbose: boolean;
    /**
     * @stability stable
     */
    protected logIgnores: boolean;
    /**
     * @stability stable
     */
    protected reports: boolean;
    /**
     * @stability stable
     */
    protected reportStacks: string[];
    /**
     * @stability stable
     */
    protected packName: string;
    /**
     * @stability stable
     */
    constructor(props?: NagPackProps);
    /**
     * @stability stable
     */
    get readPackName(): string;
    /**
     * @stability stable
     */
    get readReportStacks(): string[];
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    abstract visit(node: IConstruct): void;
    /**
     * Create a rule to be used in the NagPack.
     *
     * @param params The.
     * @stability stable
     * @IApplyRule interface with rule details.
     */
    protected applyRule(params: IApplyRule): void;
    /**
     * Check whether a specific rule should be ignored.
     *
     * @param ignores The ignores listed in cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @returns The reason the rule was ignored, or an empty string.
     * @stability stable
     */
    protected ignoreRule(ignores: NagPackSuppression[], ruleId: string): string;
    /**
     * The message to output to the console when a rule is triggered.
     *
     * @param ruleId The id of the rule.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     * @stability stable
     */
    protected createMessage(ruleId: string, info: string, explanation: string): string;
    /**
     * Write a line to the rule packs compliance report for the resource's Stack.
     *
     * @param params The.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     * @stability stable
     * @IApplyRule interface with rule details.
     */
    protected writeToStackComplianceReport(params: IApplyRule, ruleId: string, compliance: NagRuleCompliance.COMPLIANT | NagRuleCompliance.NON_COMPLIANT | 'UNKNOWN', explanation?: string): void;
    /**
     * Helper function to create a line for the compliance report.
     *
     * @param params The.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     * @stability stable
     * @IApplyRule interface with rule details.
     */
    protected createComplianceReportLine(params: IApplyRule, ruleId: string, compliance: NagRuleCompliance.COMPLIANT | NagRuleCompliance.NON_COMPLIANT | 'UNKNOWN', explanation?: string): string;
}
