"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * CloudWatch alarms have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cloudwatch_1.CfnAlarm) {
        const actionsEnabled = core_1.Stack.of(node).resolve(node.actionsEnabled);
        if (actionsEnabled === false) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        // Actions can be an array with a token that then resolves to an empty array or undefined
        const alarmActions = core_1.Stack.of(node).resolve(node.alarmActions);
        const insufficientDataActions = core_1.Stack.of(node).resolve(node.insufficientDataActions);
        const okActions = core_1.Stack.of(node).resolve(node.okActions);
        const totalAlarmActions = alarmActions ? alarmActions.length : 0;
        const totalInsufficientDataActions = insufficientDataActions
            ? insufficientDataActions.length
            : 0;
        const totalOkActions = okActions ? okActions.length : 0;
        const totalActions = totalAlarmActions + totalInsufficientDataActions + totalOkActions;
        if (totalActions == 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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