"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_backup_1 = require("@aws-cdk/aws-backup");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * DynamoDB tables are part of AWS Backup plan(s)
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_dynamodb_1.CfnTable) {
        const tableLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        const tableName = core_1.Stack.of(node).resolve(node.tableName);
        let found = false;
        for (const child of core_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_backup_1.CfnBackupSelection) {
                if (isMatchingSelection(child, tableLogicalId, tableName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the Backup Plan Selection contains the given Table
 * @param node the CfnBackupSelection to check
 * @param tableLogicalId the Cfn Logical ID of the table
 * @param tableName the name of the table
 * returns whether the CfnBackupSelection contains the given Table
 */
function isMatchingSelection(node, tableLogicalId, tableName) {
    const backupSelection = core_1.Stack.of(node).resolve(node.backupSelection);
    const resources = core_1.Stack.of(node).resolve(backupSelection.resources);
    if (Array.isArray(resources)) {
        for (const resource of resources) {
            const resolvedResource = JSON.stringify(core_1.Stack.of(node).resolve(resource));
            if (new RegExp(`${tableLogicalId}(?![\\w])`).test(resolvedResource) ||
                (tableName != undefined &&
                    new RegExp(`table\/${tableName}(?![\\w\\-_\\.])`).test(resolvedResource))) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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