"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ecr_1 = require("@aws-cdk/aws-ecr");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * ECR Repositories do not allow open access
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ecr_1.CfnRegistryPolicy) {
        const policyText = core_1.Stack.of(node).resolve(node.policyText);
        const compliant = checkStatement(policyText);
        if (compliant !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_ecr_1.CfnRepository) {
        const policyText = core_1.Stack.of(node).resolve(node.repositoryPolicyText);
        const compliant = checkStatement(policyText);
        if (compliant !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function for parsing through the policy statement
 * @param policyText the JSON policy text
 * @returns boolean
 */
function checkStatement(policyText) {
    if (policyText == undefined || policyText.Statement == undefined) {
        return true;
    }
    for (const statement of policyText.Statement) {
        const effect = statement.Effect;
        if (effect == 'Allow') {
            const awsString = statement.Principal
                ? JSON.stringify(statement.Principal)
                : '';
            if (awsString.includes('*')) {
                return false;
            }
        }
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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