"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * Containers in ECS task definitions configured for host networking have 'privileged' set to true and a non-empty non-root 'user'
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ecs_1.CfnTaskDefinition) {
        if (node.networkMode === aws_ecs_1.NetworkMode.HOST) {
            const containerDefinitions = core_1.Stack.of(node).resolve(node.containerDefinitions);
            if (containerDefinitions !== undefined) {
                for (const containerDefinition of containerDefinitions) {
                    const resolvedDefinition = core_1.Stack.of(node).resolve(containerDefinition);
                    const privileged = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedDefinition.privileged);
                    const user = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedDefinition.user);
                    if (privileged !== true || user === undefined) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    const rootIdentifiers = ['root', '0'];
                    const userParts = user.split(':');
                    for (const userPart of userParts) {
                        if (rootIdentifiers.includes(userPart.toLowerCase())) {
                            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                        }
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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