"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * IAM policies do not grant admin access
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_iam_1.CfnPolicy || node instanceof aws_iam_1.CfnManagedPolicy) {
        if (checkDocument(node, node.policyDocument)) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_iam_1.CfnGroup || node instanceof aws_iam_1.CfnRole) {
        if (node.policies != undefined && checkDocument(node, node.policies)) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function for parsing through the policy document
 * @param node the CfnResource to Check
 * @param policyDoc the JSON policy document
 * @returns boolean
 */
function checkDocument(node, policyDoc) {
    const resolvedDoc = core_1.Stack.of(node).resolve(policyDoc);
    const reg = /"Action":\[?(.*,)?"\*"(,.*)?\]?,"Effect":"Allow","Resource":\[?(.*,)?"(?:arn(?::.*(?::)?)?)?\*"(,.*)?\]?/gm;
    if (JSON.stringify(resolvedDoc).search(reg) != -1) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiSUFNUG9saWN5Tm9TdGF0ZW1lbnRzV2l0aEFkbWluQWNjZXNzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3J1bGVzL2lhbS9JQU1Qb2xpY3lOb1N0YXRlbWVudHNXaXRoQWRtaW5BY2Nlc3MudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQTs7O0VBR0U7QUFDRiwrQkFBNkI7QUFDN0IsOENBTTBCO0FBQzFCLHdDQUFtRDtBQUNuRCwrQ0FBb0Q7QUFFcEQ7OztHQUdHO0FBQ0gsa0JBQWUsTUFBTSxDQUFDLGNBQWMsQ0FDbEMsQ0FBQyxJQUFpQixFQUFxQixFQUFFO0lBQ3ZDLElBQUksSUFBSSxZQUFZLG1CQUFTLElBQUksSUFBSSxZQUFZLDBCQUFnQixFQUFFO1FBQ2pFLElBQUksYUFBYSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLEVBQUU7WUFDNUMsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7U0FDeEM7UUFDRCxPQUFPLDZCQUFpQixDQUFDLFNBQVMsQ0FBQztLQUNwQztTQUFNLElBQUksSUFBSSxZQUFZLGtCQUFRLElBQUksSUFBSSxZQUFZLGlCQUFPLEVBQUU7UUFDOUQsSUFBSSxJQUFJLENBQUMsUUFBUSxJQUFJLFNBQVMsSUFBSSxhQUFhLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUNwRSxPQUFPLDZCQUFpQixDQUFDLGFBQWEsQ0FBQztTQUN4QztRQUNELE9BQU8sNkJBQWlCLENBQUMsU0FBUyxDQUFDO0tBQ3BDO1NBQU07UUFDTCxPQUFPLDZCQUFpQixDQUFDLGNBQWMsQ0FBQztLQUN6QztBQUNILENBQUMsRUFDRCxNQUFNLEVBQ04sRUFBRSxLQUFLLEVBQUUsWUFBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUNsQyxDQUFDO0FBRUY7Ozs7O0dBS0c7QUFDSCxTQUFTLGFBQWEsQ0FBQyxJQUFpQixFQUFFLFNBQWM7SUFDdEQsTUFBTSxXQUFXLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFtQixDQUFDO0lBQ3hFLE1BQU0sR0FBRyxHQUNQLDRHQUE0RyxDQUFDO0lBQy9HLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLEVBQUU7UUFDakQsT0FBTyxJQUFJLENBQUM7S0FDYjtJQUNELE9BQU8sS0FBSyxDQUFDO0FBQ2YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG5Db3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4qL1xuaW1wb3J0IHsgcGFyc2UgfSBmcm9tICdwYXRoJztcbmltcG9ydCB7XG4gIENmblBvbGljeSxcbiAgQ2ZuTWFuYWdlZFBvbGljeSxcbiAgUG9saWN5RG9jdW1lbnQsXG4gIENmbkdyb3VwLFxuICBDZm5Sb2xlLFxufSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IENmblJlc291cmNlLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgTmFnUnVsZUNvbXBsaWFuY2UgfSBmcm9tICcuLi8uLi9uYWctcnVsZXMnO1xuXG4vKipcbiAqIElBTSBwb2xpY2llcyBkbyBub3QgZ3JhbnQgYWRtaW4gYWNjZXNzXG4gKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAqL1xuZXhwb3J0IGRlZmF1bHQgT2JqZWN0LmRlZmluZVByb3BlcnR5KFxuICAobm9kZTogQ2ZuUmVzb3VyY2UpOiBOYWdSdWxlQ29tcGxpYW5jZSA9PiB7XG4gICAgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5Qb2xpY3kgfHwgbm9kZSBpbnN0YW5jZW9mIENmbk1hbmFnZWRQb2xpY3kpIHtcbiAgICAgIGlmIChjaGVja0RvY3VtZW50KG5vZGUsIG5vZGUucG9saWN5RG9jdW1lbnQpKSB7XG4gICAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UO1xuICAgICAgfVxuICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLkNPTVBMSUFOVDtcbiAgICB9IGVsc2UgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5Hcm91cCB8fCBub2RlIGluc3RhbmNlb2YgQ2ZuUm9sZSkge1xuICAgICAgaWYgKG5vZGUucG9saWNpZXMgIT0gdW5kZWZpbmVkICYmIGNoZWNrRG9jdW1lbnQobm9kZSwgbm9kZS5wb2xpY2llcykpIHtcbiAgICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLk5PTl9DT01QTElBTlQ7XG4gICAgICB9XG4gICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuQ09NUExJQU5UO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9UX0FQUExJQ0FCTEU7XG4gICAgfVxuICB9LFxuICAnbmFtZScsXG4gIHsgdmFsdWU6IHBhcnNlKF9fZmlsZW5hbWUpLm5hbWUgfVxuKTtcblxuLyoqXG4gKiBIZWxwZXIgZnVuY3Rpb24gZm9yIHBhcnNpbmcgdGhyb3VnaCB0aGUgcG9saWN5IGRvY3VtZW50XG4gKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gQ2hlY2tcbiAqIEBwYXJhbSBwb2xpY3lEb2MgdGhlIEpTT04gcG9saWN5IGRvY3VtZW50XG4gKiBAcmV0dXJucyBib29sZWFuXG4gKi9cbmZ1bmN0aW9uIGNoZWNrRG9jdW1lbnQobm9kZTogQ2ZuUmVzb3VyY2UsIHBvbGljeURvYzogYW55KTogYm9vbGVhbiB7XG4gIGNvbnN0IHJlc29sdmVkRG9jID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShwb2xpY3lEb2MpIGFzIFBvbGljeURvY3VtZW50O1xuICBjb25zdCByZWcgPVxuICAgIC9cIkFjdGlvblwiOlxcWz8oLiosKT9cIlxcKlwiKCwuKik/XFxdPyxcIkVmZmVjdFwiOlwiQWxsb3dcIixcIlJlc291cmNlXCI6XFxbPyguKiwpP1wiKD86YXJuKD86Oi4qKD86Oik/KT8pP1xcKlwiKCwuKik/XFxdPy9nbTtcbiAgaWYgKEpTT04uc3RyaW5naWZ5KHJlc29sdmVkRG9jKS5zZWFyY2gocmVnKSAhPSAtMSkge1xuICAgIHJldHVybiB0cnVlO1xuICB9XG4gIHJldHVybiBmYWxzZTtcbn1cbiJdfQ==