"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticsearch_1 = require("@aws-cdk/aws-elasticsearch");
const aws_opensearchservice_1 = require("@aws-cdk/aws-opensearchservice");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * OpenSearch Service domains minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticsearch_1.CfnDomain || node instanceof aws_opensearchservice_1.CfnDomain) {
        const logPublishingOptions = core_1.Stack.of(node).resolve(node.logPublishingOptions);
        if (logPublishingOptions == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requiredSlowLogs = [
            logPublishingOptions === null || logPublishingOptions === void 0 ? void 0 : logPublishingOptions.SEARCH_SLOW_LOGS,
            logPublishingOptions === null || logPublishingOptions === void 0 ? void 0 : logPublishingOptions.INDEX_SLOW_LOGS,
        ];
        for (const log of requiredSlowLogs) {
            const resolvedLog = core_1.Stack.of(node).resolve(log);
            if (resolvedLog == undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
            const enabled = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedLog.enabled);
            if (!enabled) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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