"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * S3 Buckets require requests to use SSL
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_s3_1.CfnBucket) {
        const bucketLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        const bucketName = core_1.Stack.of(node).resolve(node.bucketName);
        let found = false;
        for (const child of core_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_s3_1.CfnBucketPolicy) {
                if (isMatchingCompliantPolicy(child, bucketLogicalId, bucketName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the Bucket Policy requires SSL on the given bucket.
 * @param node The CfnBucketPolicy to check.
 * @param bucketLogicalId The Cfn Logical ID of the bucket.
 * @param bucketName The name of the bucket.
 * @returns Whether the CfnBucketPolicy requires SSL on the given bucket.
 */
function isMatchingCompliantPolicy(node, bucketLogicalId, bucketName) {
    var _a, _b;
    const bucket = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.bucket);
    if (bucket !== bucketLogicalId && bucket !== bucketName) {
        return false;
    }
    const resolvedPolicyDocument = core_1.Stack.of(node).resolve(node.policyDocument);
    for (const statement of resolvedPolicyDocument.Statement) {
        const resolvedStatement = core_1.Stack.of(node).resolve(statement);
        const secureTransport = (_b = (_a = resolvedStatement === null || resolvedStatement === void 0 ? void 0 : resolvedStatement.Condition) === null || _a === void 0 ? void 0 : _a.Bool) === null || _b === void 0 ? void 0 : _b['aws:SecureTransport'];
        if (resolvedStatement.Effect === 'Deny' &&
            checkMatchingAction(resolvedStatement.Action) === true &&
            checkMatchingPrincipal(resolvedStatement.Principal) === true &&
            (secureTransport === 'false' || secureTransport === false) &&
            checkMatchingResources(node, bucketLogicalId, bucketName, resolvedStatement.Resource) === true) {
            return true;
        }
    }
    return false;
}
/**
 * Helper function to check whether the Bucket Policy applies to all actions
 * @param node The CfnBucketPolicy to check
 * @param actions The action in the bucket policy
 * @returns Whether the CfnBucketPolicy applies to all actions
 */
function checkMatchingAction(actions) {
    if (Array.isArray(actions)) {
        for (const action of actions) {
            if (action === '*' || action === 's3:*') {
                return true;
            }
        }
    }
    else if (actions === '*' || actions === 's3:*') {
        return true;
    }
    return false;
}
/**
 * Helper function to check whether the Bucket Policy applies to all principals
 * @param node The CfnBucketPolicy to check
 * @param principal The principals in the bucket policy
 * @returns Whether the CfnBucketPolicy applies to all principals
 */
function checkMatchingPrincipal(principals) {
    if (principals === '*') {
        return true;
    }
    const awsPrincipal = principals.AWS;
    if (Array.isArray(awsPrincipal)) {
        for (const account of awsPrincipal) {
            if (account === '*') {
                return true;
            }
        }
    }
    else if (awsPrincipal === '*') {
        return true;
    }
    return false;
}
/**
 * Helper function to check whether the Bucket Policy applies to the bucket and all of it's resources
 * @param node The CfnBucketPolicy to check
 * @param bucketLogicalId The Cfn Logical ID of the bucket
 * @param bucketName The name of the bucket
 * @param resources The resources in the bucket policy
 * @returns Whether the Bucket Policy applies to the bucket and all of it's resources
 */
function checkMatchingResources(node, bucketLogicalId, bucketName, resources) {
    if (!Array.isArray(resources)) {
        return false;
    }
    const bucketResourceRegexes = Array();
    const bucketObjectsRegexes = Array();
    bucketResourceRegexes.push(`(${bucketLogicalId}(?![\\w\\-]))`);
    bucketObjectsRegexes.push(`(${bucketLogicalId}(?![\\w\\-]).*\\/\\*)`);
    if (bucketName !== undefined) {
        bucketResourceRegexes.push(`(${bucketName}(?![\\w\\-]))`);
        bucketObjectsRegexes.push(`(${bucketName}(?![\\w\\-]).*\\/\\*)`);
    }
    const fullBucketResourceRegex = new RegExp(bucketResourceRegexes.join('|'));
    const fullBucketObjectsRegex = new RegExp(bucketObjectsRegexes.join('|'));
    let matchedBucketResource = false;
    let matchedObjectsResource = false;
    for (const resource of resources) {
        const resolvedResourceString = JSON.stringify(core_1.Stack.of(node).resolve(resource));
        if (matchedBucketResource === false &&
            fullBucketResourceRegex.test(resolvedResourceString) &&
            !resolvedResourceString.includes('/')) {
            matchedBucketResource = true;
        }
        else if (matchedObjectsResource === false &&
            fullBucketObjectsRegex.test(resolvedResourceString) &&
            resolvedResourceString.indexOf('/') ===
                resolvedResourceString.lastIndexOf('/')) {
            matchedObjectsResource = true;
        }
        if (matchedBucketResource === true && matchedObjectsResource === true) {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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