"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * S3 Buckets have default server-side encryption enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_s3_1.CfnBucket) {
        if (node.bucketEncryption == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const encryption = core_1.Stack.of(node).resolve(node.bucketEncryption);
        if (encryption.serverSideEncryptionConfiguration == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const sse = core_1.Stack.of(node).resolve(encryption.serverSideEncryptionConfiguration);
        for (const rule of sse) {
            const defaultEncryption = core_1.Stack.of(node).resolve(rule.serverSideEncryptionByDefault);
            if (defaultEncryption == undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
            const sseAlgorithm = nag_rules_1.NagRules.resolveIfPrimitive(node, defaultEncryption.sseAlgorithm);
            if (sseAlgorithm.toLowerCase() != 'aes256' &&
                sseAlgorithm.toLowerCase() != 'aws:kms') {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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