"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * S3 Buckets are encrypted with a KMS Key by default
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_s3_1.CfnBucket) {
        if (node.bucketEncryption == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const encryption = core_1.Stack.of(node).resolve(node.bucketEncryption);
        if (encryption.serverSideEncryptionConfiguration == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const sse = core_1.Stack.of(node).resolve(encryption.serverSideEncryptionConfiguration);
        for (const rule of sse) {
            const defaultEncryption = core_1.Stack.of(node).resolve(rule.serverSideEncryptionByDefault);
            if (defaultEncryption == undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
            const sseAlgorithm = nag_rules_1.NagRules.resolveIfPrimitive(node, defaultEncryption.sseAlgorithm);
            if (sseAlgorithm.toLowerCase() != 'aws:kms') {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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