#  Copyright (c) 2023 Roboto Technologies, Inc.
import argparse
import sys
import textwrap

from ...domain.actions import (
    Action,
    InvocationDataSourceType,
    InvocationSource,
)
from ..command import RobotoCommand
from ..context import CLIContext
from .shared import (
    ParseError,
    decorate_parser_with_compute_requirements,
    decorate_parser_with_container_parameters,
    parse_compute_requirements,
    parse_container_overrides,
)


def invoke(
    args: argparse.Namespace, context: CLIContext, parser: argparse.ArgumentParser
) -> None:
    action = Action.from_name(
        name=args.name,
        action_delegate=context.actions,
        invocation_delegate=context.invocations,
    )

    try:
        compute_requirements = parse_compute_requirements(
            args,
            default_vcpu=action.compute_requirements.vCPU,
            default_memory=action.compute_requirements.memory,
            default_storage=action.compute_requirements.storage,
        )
        container_parameters = parse_container_overrides(args)
    except ParseError as exc:
        print(exc.msg, file=sys.stderr)
        return

    invocation = action.invoke(
        input_data=args.input_data,
        data_source_id=args.dataset_id,
        data_source_type=InvocationDataSourceType.Dataset,
        invocation_source=InvocationSource.Manual,
        compute_requirement_overrides=compute_requirements,
        container_parameter_overrides=container_parameters,
    )
    print(
        f"Queued invocation of '{action.name}' action. Invocation ID: '{invocation.id}'"
    )


def invoke_parser(parser: argparse.ArgumentParser) -> None:
    parser.add_argument(
        "--name",
        required=True,
        action="store",
        help="Name of action to invoke.",
    )
    parser.add_argument(
        "--input-data",
        required=True,
        dest="input_data",
        type=str,
        nargs="+",
        action="extend",
        # fmt: off
        help=textwrap.dedent("""\
            One or many file patterns for data to download from the data source. Examples:
                front camera images, "--input-data '**/cam_front/*.jpg'";
                front and rear camera images, "--input-data '**/cam_front/*.jpg' --input-data '**/cam_rear/*.jpg'";
                all data, "--input-data '**/*'"
        """),
        # fmt: on
    )

    # As more InvocationDataSourceTypes are supported,
    # additional, mutually-exclusive argument groups should be added
    dataset_source_group = parser.add_argument_group(
        "Dataset",
        "Use a dataset as the data source for this invocation. This is currently the only option.",
    )
    dataset_source_group.add_argument(
        "--dataset-id",
        required=True,
        action="store",
        dest="dataset_id",
        help="Unique identifier for dataset to use as data source for this invocation.",
    )

    decorate_parser_with_compute_requirements(parser)
    decorate_parser_with_container_parameters(parser)


invoke_command = RobotoCommand(
    name="invoke",
    logic=invoke,
    setup_parser=invoke_parser,
    command_kwargs={"help": "Invoke an action by name."},
)
