import json
from typing import Any, Optional
import urllib.parse

from ...http import (
    ORG_OVERRIDE_HEADER,
    USER_OVERRIDE_HEADER,
    HttpClient,
)
from ...pagination import PaginatedList
from ...serde import pydantic_jsonable_dict
from ..files import FileRecord
from .delegate import (
    AccessMode,
    Credentials,
    DatasetDelegate,
)
from .http_resources import CreateDatasetRequest
from .record import (
    Administrator,
    DatasetRecord,
    StorageLocation,
)


class DatasetHttpDelegate(DatasetDelegate):
    __http_client: HttpClient
    __roboto_service_base_url: str

    def __init__(self, roboto_service_base_url: str, http_client: HttpClient) -> None:
        super().__init__()
        self.__http_client = http_client
        self.__roboto_service_base_url = roboto_service_base_url

    def headers(
        self, org_id: Optional[str], user_id: Optional[str] = None
    ) -> dict[str, str]:
        headers = {"Content-Type": "application/json"}
        if org_id:
            headers[ORG_OVERRIDE_HEADER] = org_id
        if user_id:
            headers[USER_OVERRIDE_HEADER] = user_id
        return headers

    def create_dataset(
        self,
        administrator: Administrator = Administrator.Roboto,
        metadata: Optional[dict[str, Any]] = None,
        storage_location: StorageLocation = StorageLocation.S3,
        tags: Optional[list[str]] = None,
        org_id: Optional[str] = None,
        created_by: Optional[str] = None,
    ) -> DatasetRecord:
        """
        Create a new dataset.
        """
        url = f"{self.__roboto_service_base_url}/v1/datasets"
        request_body = CreateDatasetRequest(
            administrator=administrator,
            metadata=metadata if metadata is not None else {},
            storage_location=storage_location,
            tags=tags if tags is not None else [],
        )
        response = self.__http_client.post(
            url,
            data=pydantic_jsonable_dict(request_body),
            headers=self.headers(org_id, created_by),
        )
        return DatasetRecord.parse_obj(response.from_json(json_path=["data"]))

    def get_dataset_by_primary_key(
        self,
        dataset_id: str,
        org_id: Optional[str] = None,
    ) -> DatasetRecord:
        """
        Get a dataset by its primary key (org_id, dataset_id)
        """
        url = f"{self.__roboto_service_base_url}/v1/datasets/{dataset_id}"
        response = self.__http_client.get(url, headers=self.headers(org_id))
        return DatasetRecord.parse_obj(response.from_json(json_path=["data"]))

    def get_temporary_credentials(
        self, record: DatasetRecord, mode: AccessMode, caller: Optional[str] = None
    ) -> Credentials:
        """
        Get temporary credentials to access a dataset.
        """
        query_params = {"mode": mode.value}
        encoded_qs = urllib.parse.urlencode(query_params)
        url = f"{self.__roboto_service_base_url}/v1/datasets/{record.dataset_id}/credentials?{encoded_qs}"
        response = self.__http_client.get(url, self.headers(record.org_id, caller))
        return Credentials.parse_obj(response.from_json(json_path=["data"]))

    def list_files(
        self,
        dataset_id: str,
        org_id: Optional[str] = None,
        page_token: Optional[dict[str, str]] = None,
    ) -> PaginatedList[FileRecord]:
        """
        List files associated with dataset.

        Files are associated with datasets in an eventually-consistent manner,
        so there will likely be delay between a file being uploaded and it appearing in this list.
        """
        url = f"{self.__roboto_service_base_url}/v1/datasets/{dataset_id}/files"
        if page_token:
            query_str = urllib.parse.urlencode(page_token)
            url = f"{url}?{query_str}"
        response = self.__http_client.get(url, headers=self.headers(org_id))
        unmarshalled = response.from_json(json_path=["data"])
        return PaginatedList(
            items=[FileRecord.parse_obj(file) for file in unmarshalled["items"]],
            next_token=unmarshalled["next_token"],
        )

    def query_datasets(
        self,
        filters: dict[str, Any],
        org_id: Optional[str] = None,
        page_token: Optional[dict[str, str]] = None,
    ) -> PaginatedList[DatasetRecord]:
        if page_token:
            filters["page_token"] = page_token

        safe_filters = json.loads(json.dumps(filters))
        url = f"{self.__roboto_service_base_url}/v1/datasets/query"
        res = self.__http_client.post(
            url, data=safe_filters, headers=self.headers(org_id)
        )
        unmarshalled = res.from_json(json_path=["data"])
        return PaginatedList(
            items=[
                DatasetRecord.parse_obj(dataset) for dataset in unmarshalled["items"]
            ],
            next_token=unmarshalled["next_token"],
        )
