#!/usr/bin/env python

"""

This is the script to generate the marine heatwave analysis
including the statistic of climatology and percentile threshold
based on the forecast(reforecast) and historical run 
that is generated by Andrew Ross at GFDL.

"""
import warnings
import xarray as xr
from mom6.mom6_module.mom6_statistics import (
    ForecastClimatology,
    ForecastQuantile,
    CoordinateWrangle
)
from mom6.mom6_module.mom6_types import (
    TimeGroupByOptions
)
from mom6.mom6_module.mom6_detrend import ForecastDetrend

warnings.simplefilter("ignore")
xr.set_options(keep_attrs=True)


class MarineHeatwaveForecast:
    """
    Class for calculating the marine heatwave related
    statistics and the resulted identified event
    and the probability with the use of prob forecast
    provided by the forecast/reforecast

    The method should be able to accommodate the 
    raw and regridded format

    Parameters
    ----------
    ds_data : xr.Dataset
        The sea surface temperature dataset one want to use to 
        derived the marine heatwave. The coordinate
        must have the name "lon" and "lat" exactly (raw grid
        geolon and geolat need to be renamed).
    sst_name : str
        The sea surface temperature variable name in the dataset
    initialization_name : str, optional
        initialization dimension name, by default 'init'
    member_name : str, optional
        ensemble member dimension name, by default 'member'
    time_frequency : TimeGroupByOptions, optional
        name in time frequency to do the time group, by default 'month'
        'year', 'month', 'dayofyear' are the available options.
    """

    def __init__(
        self,
        ds_data : xr.Dataset,
        sst_name : str = 'tos',
        initialization_name : str = 'init',
        member_name : str = 'member',
        time_frequency : TimeGroupByOptions = 'month'
    ) -> None:

        self.dataset = CoordinateWrangle(ds_data).to_360()
        self.varname = sst_name
        self.init = initialization_name
        self.mem = member_name
        self.tfreq = time_frequency

    def generate_forecast_batch(
        self,
        climo_start_year : int = 1993,
        climo_end_year : int = 2020,
        anom_start_year : int = 1993,
        anom_end_year : int = 2020,
        quantile_threshold : float = 90.,
        detrend : bool = False
    ) -> xr.Dataset:
        """generate the MHW statistics and identify MHW

        Parameters
        ----------
        climo_start_year : int, optional
            start year of climatology and threshold period, by default 1993
        climo_end_year : int, optional
            end year of climatology and threshold period, by default 2020
        anom_start_year : int, optional
            start year of anomaly that need to identify MHW, by default 1993
        anom_end_year : int, optional
            end year of anomaly that need to identify MHW, by default 2020
        quantile_threshold : float, optional
            quantile value that define the threshold, by default 90.
        detrend : bool, optional
            flag for whether the MHW is based on detrended ssta or not.

        Returns
        -------
        xr.Dataset
            The dataset including MHW probability, Magnitude, threshold.
        """

        # calculate anomaly based on climatology
        class_forecast_climo = ForecastClimatology(self.dataset,self.varname)
        dict_anom_thres = class_forecast_climo.generate_anom_batch(
            climo_start_year,
            climo_end_year,
            climo_start_year, # force the anom start year for threshold be the same as climo period
            climo_end_year,   # force the anom end year for threshold be the same as climo period
            'persist'
        )

        # detrend or not
        if detrend:
            class_detrend_thres = ForecastDetrend(dict_anom_thres['anomaly'])
            dict_anom_thres['anomaly'], ds_p = class_detrend_thres.detrend_linear(
                precompute_coeff=False,
                in_place_memory_replace=True
            )

        # anomaly used for the threshold
        ds_anom_thres = xr.Dataset()
        ds_anom_thres[f'{self.varname}_anom'] = dict_anom_thres['anomaly']
        ds_anom_thres['lon'] = self.dataset['lon']
        ds_anom_thres['lat'] = self.dataset['lat']

        # calculate threshold
        # if detrend:
        #     ### in memery result when creating the class
        #     class_forecast_quantile = ForecastQuantile(
        #         ds_anom_thres.compute(),
        #         f'{self.varname}_anom'
        #     )
        #     da_threshold = class_forecast_quantile.generate_quantile(
        #         climo_start_year,
        #         climo_end_year,
        #         quantile_threshold,
        #         dask_obj=False
        #     )
        # else:
        class_forecast_quantile = ForecastQuantile(ds_anom_thres,f'{self.varname}_anom')
        ### in memery result not lazy-loaded (same as climo period)
        da_threshold = class_forecast_quantile.generate_quantile(
            climo_start_year,
            climo_end_year,
            quantile_threshold,
            dask_obj=True
        )

        # anomaly that need to find MHW
        dict_anom = class_forecast_climo.generate_anom_batch(
            climo_start_year,
            climo_end_year,
            anom_start_year,
            anom_end_year,
            'persist',
            precompute_climo = True,
            da_climo = dict_anom_thres['climatology']
        )
        da_anom = dict_anom['anomaly']

        if detrend:
            class_detrend = ForecastDetrend(da_anom)
            da_anom,_ = class_detrend.detrend_linear(
                precompute_coeff=True,
                ds_coeff=ds_p,
                in_place_memory_replace=True
            )

        # calculate average mhw magnitude
        da_mhw_mag = da_anom.where(da_anom.groupby(f'{self.init}.{self.tfreq}')>=da_threshold)
        da_mhw_mag_ave = da_anom.mean(dim=f'{self.mem}').compute()
        da_mhw_mag_ave.attrs['mhw_magnitude_definition'] = 'ensemble mean of all sst anomaly'

        # calculate mhw event
        da_mhw = (
            da_mhw_mag
            .where(da_mhw_mag.isnull(),other=1)
            .sum(dim=f'{self.mem}',skipna=True)
        )

        # calculate probability
        da_mask = da_anom.where(da_anom.isnull(),other=1.)
        da_event = da_mask.sum(dim=f'{self.mem}')
        da_prob = (da_mhw/da_event).compute()

        # output dataset
        ds_mhw = xr.Dataset()
        if detrend :
            ds_mhw['polyfit_coefficients'] = ds_p['polyfit_coefficients']

        ds_mhw[f'{self.varname}_threshold{quantile_threshold:02d}'] = da_threshold
        ds_mhw[f'{self.varname}_threshold{quantile_threshold:02d}'].attrs['long_name'] = (
            f'{self.varname} threshold{quantile_threshold:02d}'
        )
        ds_mhw[f'{self.varname}_threshold{quantile_threshold:02d}'].attrs['units'] = 'degC'

        ds_mhw[f'{self.varname}_climo'] = dict_anom_thres['climatology']
        ds_mhw[f'{self.varname}_climo'].attrs['long_name'] = (
            f'{self.varname} climatology'
        )
        ds_mhw[f'{self.varname}_climo'].attrs['units'] = 'degC'

        ds_mhw[f'mhw_prob{quantile_threshold:02d}'] = da_prob
        ds_mhw[f'mhw_prob{quantile_threshold:02d}'].attrs['long_name'] = (
            f'marine heatwave probability (threshold{quantile_threshold:02d})'
        )
        ds_mhw[f'mhw_prob{quantile_threshold:02d}'].attrs['units'] = 'unitless'

        ds_mhw['ssta_avg'] = da_mhw_mag_ave
        ds_mhw['ssta_avg'].attrs['long_name'] = (
            'anomalous sea surface temperature ensemble mean'
        )
        ds_mhw['ssta_avg'].attrs['units'] = 'degC'

        ds_mhw['ssta'] = da_anom
        ds_mhw['ssta'].attrs['long_name'] = (
            'anomalous sea surface temperature'
        )
        ds_mhw['ssta'].attrs['units'] = 'degC'

        ds_mhw.attrs['period_of_quantile'] = da_threshold.attrs['period_of_quantile']
        ds_mhw.attrs['period_of_climatology'] = da_threshold.attrs['period_of_climatology']

        return ds_mhw

    def generate_forecast_single(
        self,
        init_time : str = '2022-03',
        da_climo : xr.DataArray = None,
        da_threshold : xr.DataArray = None
    ) -> xr.Dataset:
        """generate MHW forecast for single initialization
        based on existing stats (climatology and threshold)

        Parameters
        ----------
        init_time : str, optional
            the signal initialization time in the format of YYYY-MM, by default '2022-03'
        da_climo : xr.DataArray, optional
            the dataarray containing the climatology data, by default None
        da_threshold : xr.DataArray, optional
            the dataarray containing the threshold data, by default None

        Returns
        -------
        xr.Dataset
            The dataset including MHW probability, Magnitude, threshold.
        """

        # crop data
        da_data = self.dataset[self.varname].sel(
            {self.init :init_time}
        ).load()

        # test if the da_data crop period exist
        if len(da_data[self.init].data) == 0:
            raise ValueError(
                "The data array is empty based on the kwarg "+
                "init_time"
            )

        # calculate anomaly based on climatology
        da_anom = (
            da_data.groupby(f'{self.init}.{self.tfreq}')
            - da_climo
        ).compute()

        # calculate average mhw magnitude
        da_mhw_mag = da_anom.where(da_anom.groupby(f'{self.init}.{self.tfreq}')>=da_threshold)
        da_mhw_mag_ave = da_anom.mean(dim=f'{self.mem}').compute()
        da_mhw_mag_ave.attrs['mhw_magnitude_definition'] = 'ensemble mean of all sst anomaly'

        # calculate mhw event
        da_mhw = (
            da_mhw_mag
            .where(da_mhw_mag.isnull(),other=1)
            .sum(dim=f'{self.mem}',skipna=True)
        )

        # calculate probability
        da_mask = da_anom.where(da_anom.isnull(),other=1.)
        da_event = da_mask.sum(dim=f'{self.mem}')
        da_prob = (da_mhw/da_event).compute()

        # output dataset
        ds_mhw = xr.Dataset()
        try:
            ds_mhw.attrs['period_of_quantile'] = da_threshold.attrs['period_of_quantile']
            ds_mhw.attrs['period_of_climatology'] = da_climo.attrs['period_of_climatology']
            quantile_threshold = int(da_threshold.attrs['period_of_quantile'].split()[1])
        except KeyError as e:
            raise AttributeError(
                'quantile file is not standard file that provide quantile number'
            ) from e
        except ValueError as e:
            raise ValueError(
                'The "quantile_threshold" dataarray.... does not have period_of_quantile attrs'
            ) from e

        ds_mhw[f'mhw_prob{quantile_threshold:02d}'] = da_prob
        ds_mhw['ssta_avg'] = da_mhw_mag_ave
        ds_mhw['mhw_mag_indentified_ens'] = da_mhw_mag

        ds_mhw[f'mhw_prob{quantile_threshold:02d}'].attrs['long_name'] = (
            f'marine heatwave probability (threshold{quantile_threshold:02d})'
        )
        ds_mhw[f'mhw_prob{quantile_threshold:02d}'].attrs['units'] = 'unitless'

        ds_mhw['ssta_avg'].attrs['long_name'] = (
            'anomalous sea surface temperature ensemble mean'
        )
        ds_mhw['ssta_avg'].attrs['units'] = 'degC'

        ds_mhw['mhw_mag_indentified_ens'].attrs['long_name'] = (
            'marine heatwave magnitude in each ensemble'
        )
        ds_mhw['mhw_mag_indentified_ens'].attrs['units'] = 'degC'

        return ds_mhw
