#!/usr/bin/env python

"""

This is the script to generate the general statistics of all field
based on the forecast(reforecast) and historical run 
that is generated by Andrew Ross at GFDL.

"""
import numpy as np
import xarray as xr
import dask
from mom6.mom6_module import time_series_processes as tsp
from mom6.mom6_module.mom6_types import (
    TimeGroupByOptions,
    DaskOptions
)

xr.set_options(keep_attrs=True)

class CoordinateWrangle:
    """
    Class to make sure the coordinate is consistent
    in raw grid and regridded data.

    The coordinate
    must have the name "lon" and "lat" exactly (raw grid
    geolon and geolat need to be renamed).

    Parameters
    ----------
    ds_data : xr.Dataset
        The dataset one want to use to
        derive the forecast statistics.

    """
    def __init__(
       self,
       ds_data : xr.Dataset,
    ) -> None:

        self.dataset = ds_data.copy(deep=True)

    def check_coord_name(self):
        """check coordinate name

        Raises
        ------
        KeyError
            standard output to let user know the exact coord
            name should be 'lon' and 'lat' if not matching.
        """
        # check lon name
        try:
            self.dataset['lon'].data
        except KeyError as e:
            raise KeyError(
                "Coordinates should have 'lon' with exact naming"
            ) from e

        # check lat name
        try:
            self.dataset['lat'].data
        except KeyError as e:
            raise KeyError(
                "Coordinates should have 'lat' with exact naming"
            ) from e

    def to_360(self):
        """check coord name and change the coordinate from -180-180
        to 0-360

        Returns
        -------
        xr.Dataset
            the modified deep copied dataset
        """
        # check coord name first
        self.check_coord_name()

        # change longitude range from -180 180 to 0 360
        lon = self.dataset['lon'].data
        if np.min(lon)<0:
            # perform where only if needed
            lon_ind = np.where(lon<0)
            lon[lon_ind] += 360.
            self.dataset['lon'].data = lon

        return self.dataset

class ForecastClimatology:
    """
    Class for calculating the climatology of forecast

    Parameters
    ----------
    ds_data : xr.Dataset
        The dataset one want to use to 
        derive the forecast statistics.
    var_name : str
        The variable name in the dataset 
    initialization_name : str, optional
        initialization dimension name, by default 'init'
    member_name : str, optional
        ensemble member dimension name, by default 'member'
    time_frequency : TimeGroupByOptions, optional
        name in time frequency to do the time group, by default 'month'
        'year', 'month', 'dayofyear' are the available options.

    """

    def __init__(
        self,
        ds_data : xr.Dataset,
        var_name : str,
        initialization_name : str = 'init',
        member_name : str = 'member',
        time_frequency : TimeGroupByOptions = 'month'
    ) -> None:

        # self.dataset = CoordinateWrangle(ds_data).to_360()
        self.dataset = ds_data
        self.varname = var_name
        self.init = initialization_name
        self.mem = member_name
        self.tfreq = time_frequency

    def generate_climo(
        self,
        climo_start_year : int = 1993,
        climo_end_year : int = 2020,
        dask_option : DaskOptions = 'lazy'
    ) -> xr.DataArray:
        """Generate the climatology based on the input 
        dataset covered period

        Parameters
        ----------
        climo_start_year : int, optional
            start year to calculation the climatology, by default 1993
        climo_end_year : int, optional
            end year to calculation the climatology, by default 2020
        dask_option : DaskOptions, optional
            flag to determine one want the return result
            to be 'compute', 'persist' or keep 'lazy', by default 'lazy'
        
        Returns
        -------
        xr.DataArray
            dataarray which represent the climatology

        Raises
        ------
        ValueError
            when the kwarg climo_start_year & climo_end_year result in 
            empty array crop
        """
        # getting the dataset
        ds_data = self.dataset

        # crop data
        da_data = ds_data[self.varname].sel(
            {self.init :
                slice(
                f'{climo_start_year}-01',
                f'{climo_end_year}-12')
            }
        )

        # test if the da_data crop period exist
        if len(da_data[self.init].data) == 0:
            raise ValueError(
                "The data array is empty based on the kwarg "+
                "climo_start_year & climo_end_year"
            )

        if self.tfreq != 'dayofyear':
            da_climo = (
                da_data
                .mean(dim=f'{self.mem}')
                .groupby(f'{self.init}.{self.tfreq}')
                .mean(dim=f'{self.init}')
            )
        else :
            da_climo = tsp.cal_daily_climo(
                da_data.mean(dim=f'{self.mem}'),
                smooth = True,
                dim = self.init,
                nharm = 4,
                apply_taper = False
            )


        da_climo.attrs['period_of_climatology'] = f'year {climo_start_year} to {climo_end_year}'

        if dask_option == 'lazy' :
            return da_climo
        elif dask_option == 'persist':
            return da_climo.persist()
        elif dask_option == 'compute':
            return da_climo.compute()

    def generate_anom_batch(
        self,
        climo_start_year : int = 1993,
        climo_end_year : int = 2020,
        anom_start_year : int = 1993,
        anom_end_year : int = 2020,
        dask_option : DaskOptions = 'lazy',
        precompute_climo : bool = False,
        da_climo : xr.DataArray = None
    ) -> dict:
        """Generate the anomaly based on the input 
        dataset covered period

        Parameters
        ----------
        climo_start_year : int, optional
            start year to calculation the climatology, by default 1993
        climo_end_year : int, optional
            end year to calculation the climatology, by default 2020
        anom_start_year : int, optional
            start year to calculation the anomaly, by default 1993
        anom_end_year : int, optional
            end year to calculation the anomaly, by default 2020
        dask_option : DaskOptions, optional
            flag to determine one want the return result
            to be 'compute', 'persist' or keep 'lazy' in anomaly, by default 'lazy'
        precompute_climo : bool, optional
            If a pre-computed climatology dataset is provided, by default False
        da_climo : xr.DataArray, optional
            pre-computed climatology dataset, by default None

        Returns
        -------
        dict
            anomaly: dataarray which represent the anomaly,
            climatology: dataarray which represent the climatology
        
        Raises
        ------
        ValueError
            when the kwarg anom_start_year & anom_end_year result in 
            empty array crop
        ValueError
            when the precomputed da_climo does not have the desired attrs
        """

        # getting the dataset
        ds_data = self.dataset

        if precompute_climo:
            try:
                period_string = da_climo.attrs['period_of_climatology']
            except KeyError as e:
                raise ValueError(
                    'input climatology dataarray does not have '+
                    'attribute name "period_of_climatology" in '+
                    'the format of year "climo_start_year" to "climo_end_year"'
                ) from e

            period_string_split = period_string.split()
            # reassigning the start and end based on the climo data
            climo_start_year = int(period_string_split[1])
            climo_end_year = int(period_string_split[3])

        else:
            # getting the climatology
            da_climo = self.generate_climo(
                climo_start_year,
                climo_end_year,
                dask_option='compute'
            )

        da_data = (
            ds_data[self.varname]
            .sel({self.init :
                slice(
                f'{anom_start_year}-01',
                f'{anom_end_year}-12')
            })
        )

        # test if the da_data crop period exist
        if len(da_data[self.init].data)==0:
            raise ValueError(
                "The data array is empty based on the kwarg "+
                "anom_start_year & anom_end_year"
            )

        da_anom = (
            da_data.groupby(f'{self.init}.{self.tfreq}')
            - da_climo
        )

        da_anom.attrs['period_of_climatology'] = f'year {climo_start_year} to {climo_end_year}'

        if dask_option == 'lazy' :
            return {'anomaly':da_anom,'climatology':da_climo}
        elif dask_option == 'persist':
            return {'anomaly':da_anom.persist(),'climatology':da_climo}
        elif dask_option == 'compute':
            return {'anomaly':da_anom.compute(),'climatology':da_climo}

class ForecastQuantile:
    """
    Class for calculating the quantile of forecast

    The method should be able to accommodate the 
    raw and regridded format

    Parameters
    ----------
    ds_data : xr.Dataset
        The dataset one want to use to 
        derive the forecast statistics.
    var_name : str
        The variable name in the dataset 
    initialization_name : str, optional
        initialization dimension name, by default 'init'
    member_name : str, optional
        ensemble member dimension name, by default 'member'
    time_frequency : TimeGroupByOptions, optional
        name in time frequency to do the time group, by default 'month'
        'year', 'month', 'dayofyear' are the available options.
    """
    def __init__(
        self,
        ds_data : xr.Dataset,
        var_name : str,
        initialization_name : str = 'init',
        member_name : str = 'member',
        time_frequency : TimeGroupByOptions = 'month'
    ) -> None:
        # self.dataset = CoordinateWrangle(ds_data).to_360()
        self.dataset = ds_data
        self.varname = var_name
        self.init = initialization_name
        self.mem = member_name
        self.tfreq = time_frequency

    def generate_quantile(
        self,
        quantile_start_year : int = 1993,
        quantile_end_year : int = 2020,
        quantile_threshold : float = 90.,
        dask_obj : bool = True
    ) -> xr.DataArray:
        """Generate the quantile based on the input 
        dataset covered period. The output will be 
        loaded to the memory (not lazy-loaded).

        Parameters
        ----------
        quantile_start_year : int, optional
            start year for the period of determining the quantile, by default 1993
        quantile_end_year : int, optional
            end year for the period of determining the quantile, by default 2020
        quantile_threshold : float, optional
            quantile value that define the threshold, by default 90.
        dask_obj : bool, optional
            if the input dataset is a dask object or not

        Returns
        -------
        xr.DataArray
            dataarry which represent the quantile value. 
            value loaded in the memory.

        Raises
        ------
        ValueError
            when the kwarg quantile_start_year & quantile_end_year result in 
            empty array crop
        """

        # getting the dataset period
        da_data = (
            self.dataset[self.varname]
            .sel({
                self.init:slice(
                f'{quantile_start_year}-01',
                f'{quantile_end_year}-12'
                )
            })
        )

        # test if the da_data crop period exist
        if len(da_data[self.init].data) == 0:
            raise ValueError(
                "The data array is empty based on the kwarg "+
                "quantile_start_year & quantile_end_year"
            )

        # getting unique initialized time for threshold statistic
        init_freq = np.unique(
            da_data[f'{self.init}.{self.tfreq}'].data
        )

        result_list = []
        if dask_obj:
            for init in init_freq:
                result_list.append(
                    da_data
                    .where(
                        da_data[f'{self.init}.{self.tfreq}'] == init,
                        drop=True
                    )
                    .stack(allens=(self.init,self.mem))
                    .chunk({"allens":-1})
                    .quantile(
                        quantile_threshold*0.01,
                        dim = 'allens',
                        method = 'linear',
                        skipna = True
                    )
                )
            da_threshold_list = dask.compute(*result_list)
        else:
            for init in init_freq:
                result_list.append(
                    da_data
                    .where(
                        da_data[f'{self.init}.{self.tfreq}'] == init,
                        drop=True
                    )
                    .stack(allens=(self.init,self.mem))
                    .quantile(
                        quantile_threshold*0.01,
                        dim = 'allens',
                        method = 'linear',
                        skipna = True
                    )
                )
            da_threshold_list = result_list
        da_threshold = xr.concat(da_threshold_list, dim=self.tfreq)
        da_threshold[self.tfreq] = init_freq

        da_threshold.attrs['period_of_quantile'] = (
            f'The {quantile_threshold} quantile from '+
            f'year {quantile_start_year} to {quantile_end_year}'
        )

        return da_threshold


class HindcastClimatology:
    """
    Class for calculating the climatology of hindcast

    Parameters
    ----------
    ds_data : xr.Dataset
        The dataset one want to use to 
        derive the historical run statistics.
    var_name : str
        The variable name in the dataset 
    initialization_name : str, optional
        initialization dimension name, by default 'init'
    member_name : str, optional
        ensemble member dimension name, by default 'member'
    time_frequency : TimeGroupByOptions, optional
        name in time frequency to do the time group, by default 'month'
        'year', 'month', 'dayofyear' are the available options.

    """

    def __init__(
        self,
        ds_data : xr.Dataset,
        var_name : str,
        time_name : str = 'time',
        time_frequency : TimeGroupByOptions = 'month'
    ) -> None:
        # self.dataset = CoordinateWrangle(ds_data).to_360()
        self.dataset = ds_data
        self.varname = var_name
        self.timename = time_name
        self.tfreq = time_frequency

    def generate_climo(
        self,
        climo_start_year : int = 1993,
        climo_end_year : int = 2020,
        dask_option : DaskOptions = 'lazy'
    ) -> xr.DataArray:
        """Generate the climatology based on the input 
        dataset covered period

        Parameters
        ----------
        climo_start_year : int, optional
            start year to calculation the climatology, by default 1993
        climo_end_year : int, optional
            end year to calculation the climatology, by default 2020
        dask_option : DaskOptions, optional
            flag to determine one want the return result
            to be 'compute', 'persist' or keep 'lazy', by default 'lazy'
        
        Returns
        -------
        xr.DataArray
            dataarray which represent the climatology

        Raises
        ------
        ValueError
            when the kwarg climo_start_year & climo_end_year result in 
            empty array crop
        """
        # getting the dataset
        ds_data = self.dataset

        # crop data
        da_data = ds_data[self.varname].sel(
            {self.timename :
                slice(
                f'{climo_start_year}-01',
                f'{climo_end_year}-12')
            }
        )

        # test if the da_data crop period exist
        if len(da_data[self.timename].data) == 0:
            raise ValueError(
                "The data array is empty based on the kwarg "+
                "climo_start_year & climo_end_year"
            )

        # monthly and daily implementation
        if self.tfreq != 'dayofyear':
            da_climo = (
                da_data
                .groupby(f'{self.timename}.{self.tfreq}')
                .mean(dim=f'{self.timename}')
            )
        else :
            da_climo = tsp.cal_daily_climo(
                da_data,
                smooth = True,
                dim = self.timename,
                nharm = 4,
                apply_taper = False
            )
            print('forced compute due to fft in 4 harmonic calculation')

        da_climo.attrs['period_of_climatology'] = f'year {climo_start_year} to {climo_end_year}'

        if dask_option == 'lazy' :
            return da_climo
        elif dask_option == 'persist':
            return da_climo.persist()
        elif dask_option == 'compute':
            return da_climo.compute()

    def generate_anom_batch(
        self,
        climo_start_year : int = 1993,
        climo_end_year : int = 2020,
        anom_start_year : int = 1993,
        anom_end_year : int = 2020,
        dask_option : DaskOptions = 'lazy',
        precompute_climo : bool = False,
        da_climo : xr.DataArray = None
    ) -> dict:
        """Generate the anomaly based on the input 
        dataset covered period

        Parameters
        ----------
        climo_start_year : int, optional
            start year to calculation the climatology, by default 1993
        climo_end_year : int, optional
            end year to calculation the climatology, by default 2020
        dask_option : DaskOptions, optional
            flag to determine one want the return result
            to be 'compute', 'persist' or keep 'lazy' in anomaly, by default 'lazy'

        Returns
        -------
        dict
            anomaly: dataarray which represent the anomaly,
            climatology: dataarray which represent the climatology
        
        Raises
        ------
        ValueError
            when the kwarg anom_start_year & anom_end_year result in 
            empty array crop

        Parameters
        ----------
        climo_start_year : int, optional
            start year to calculation the climatology, by default 1993
        climo_end_year : int, optional
            end year to calculation the climatology, by default 2020
        anom_start_year : int, optional
            start year to calculation the anomaly, by default 1993
        anom_end_year : int, optional
            end year to calculation the anomaly, by default 2020
        dask_option : DaskOptions, optional
            flag to determine one want the return result
            to be 'compute', 'persist' or keep 'lazy' in anomaly, by default 'lazy'
        precompute_climo : bool, optional
            If a pre-computed climatology dataset is provided, by default False
        da_climo : xr.DataArray, optional
            pre-computed climatology dataset, by default None

        Returns
        -------
        dict
            anomaly: dataarray which represent the anomaly,
            climatology: dataarray which represent the climatology
        
        Raises
        ------
        ValueError
            when the kwarg anom_start_year & anom_end_year result in 
            empty array crop
        ValueError
            when the precomputed da_climo does not have the desired attrs
        """

        # getting the dataset
        ds_data = self.dataset

        if precompute_climo:
            try:
                period_string = da_climo.attrs['period_of_climatology']
            except KeyError as e:
                raise ValueError(
                    'input climatology dataarray does not have '+
                    'attribute name "period_of_climatology" in '+
                    'the format of year "climo_start_year" to "climo_end_year"'
                ) from e

            period_string_split = period_string.split()
            # reassigning the start and end based on the climo data
            climo_start_year = int(period_string_split[1])
            climo_end_year = int(period_string_split[3])

        else:
            # getting the climatology
            da_climo = self.generate_climo(
                climo_start_year,
                climo_end_year,
                dask_option='compute'
            )

        # suitbale for monthly and daily implementation
        da_data = (
            ds_data[self.varname]
            .sel({self.timename :
                slice(
                f'{anom_start_year}-01',
                f'{anom_end_year}-12')
            })
        )

        # test if the da_data crop period exist
        if len(da_data[self.timename].data)==0:
            raise ValueError(
                "The data array is empty based on the kwarg "+
                "anom_start_year & anom_end_year"
            )

        da_anom = (
            da_data.groupby(f'{self.timename}.{self.tfreq}')
            - da_climo
        )

        da_anom.attrs['period_of_climatology'] = f'year {climo_start_year} to {climo_end_year}'

        if dask_option == 'lazy' :
            return {'anomaly':da_anom,'climatology':da_climo}
        elif dask_option == 'persist':
            return {'anomaly':da_anom.persist(),'climatology':da_climo}
        elif dask_option == 'compute':
            return {'anomaly':da_anom.compute(),'climatology':da_climo}


class HistoricalQuantile:
    """
    Class for calculating the quantile of historical

    The method should be able to accommodate the 
    raw and regridded format

    Parameters
    ----------
    ds_data : xr.Dataset
        The dataset one want to use to 
        derive the historical run statistics.
    var_name : str
        The variable name in the dataset 
    initialization_name : str, optional
        initialization dimension name, by default 'init'
    member_name : str, optional
        ensemble member dimension name, by default 'member'
    time_frequency : TimeGroupByOptions, optional
        name in time frequency to do the time group, by default 'month'
        'year', 'month', 'dayofyear' are the available options.
    """
    def __init__(
        self,
        ds_data : xr.Dataset,
        var_name : str,
        time_name : str = 'time',
        time_frequency : TimeGroupByOptions = 'month'
    ) -> None:

        # self.dataset = CoordinateWrangle(ds_data).to_360()
        self.dataset = ds_data
        self.varname = var_name
        self.timename = time_name
        self.tfreq = time_frequency

    def generate_quantile(
        self,
        quantile_start_year : int = 1993,
        quantile_end_year : int = 2020,
        quantile_threshold : float = 90.,
        dask_obj : bool = True
    ) -> xr.DataArray:
        """Generate the quantile based on the input 
        dataset covered period. The output will be 
        loaded to the memory (not lazy-loaded).

        Parameters
        ----------
        quantile_start_year : int, optional
            start year for the period of determining the quantile, by default 1993
        quantile_end_year : int, optional
            end year for the period of determining the quantile, by default 2020
        quantile_threshold : float, optional
            quantile value that define the threshold, by default 90.
        dask_obj : bool, optional
            if the input dataset is a dask object or not

        Returns
        -------
        xr.DataArray
            dataarry which represent the quantile value.
            value loaded in the memory.

        Raises
        ------
        ValueError
            when the kwarg quantile_start_year & quantile_end_year result in 
            empty array crop
        """

        # getting the dataset period
        da_data = (
            self.dataset[self.varname]
            .sel({
                self.timename:slice(
                f'{quantile_start_year}-01',
                f'{quantile_end_year}-12'
                )
            })
        )

        # test if the da_data crop period exist
        if len(da_data[self.timename].data) == 0:
            raise ValueError(
                "The data array is empty based on the kwarg "+
                "quantile_start_year & quantile_end_year"
            )

        # getting unique time for threshold statistic
        time_group = np.unique(
            da_data[f'{self.timename}.{self.tfreq}'].data
        )

        result_list = []
        if dask_obj:
            for time in time_group:
                result_list.append(
                    da_data
                    .where(
                        da_data[f'{self.timename}.{self.tfreq}'] == time,
                        drop=True
                    )
                    .chunk({self.timename:-1})
                    .quantile(
                        quantile_threshold*0.01,
                        dim = self.timename,
                        method = 'linear',
                        skipna = True
                    )
                )
            da_threshold_list = dask.compute(*result_list)
        else:
            for time in time_group:
                result_list.append(
                    da_data
                    .where(
                        da_data[f'{self.timename}.{self.tfreq}'] == time,
                        drop=True
                    )
                    .quantile(
                        quantile_threshold*0.01,
                        dim = self.timename,
                        method = 'linear',
                        skipna = True
                    )
                )
            da_threshold_list = result_list

        # combined the threshold for different groupby time
        da_threshold = xr.concat(da_threshold_list, dim=self.tfreq)
        da_threshold[self.tfreq] = time_group

        da_threshold.attrs['period_of_quantile'] = (
            f'The {quantile_threshold} quantile from '+
            f'year {quantile_start_year} to {quantile_end_year}'
        )

        return da_threshold
