"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeliveryStream = exports.StreamEncryption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const cdk = require("aws-cdk-lib");
const region_info_1 = require("aws-cdk-lib/region-info");
const constructs_1 = require("constructs");
const kinesisfirehose_canned_metrics_generated_1 = require("./kinesisfirehose-canned-metrics.generated");
const aws_kinesisfirehose_1 = require("aws-cdk-lib/aws-kinesisfirehose");
const PUT_RECORD_ACTIONS = [
    'firehose:PutRecord',
    'firehose:PutRecordBatch',
];
/**
 * Base class for new and imported Kinesis Data Firehose delivery streams.
 */
class DeliveryStreamBase extends cdk.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.connections = setConnections(this);
    }
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            resourceArns: [this.deliveryStreamArn],
            grantee: grantee,
            actions: actions,
        });
    }
    grantPutRecords(grantee) {
        return this.grant(grantee, ...PUT_RECORD_ACTIONS);
    }
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Firehose',
            metricName: metricName,
            dimensionsMap: {
                DeliveryStreamName: this.deliveryStreamName,
            },
            ...props,
        }).attachTo(this);
    }
    metricIncomingBytes(props) {
        return this.cannedMetric(kinesisfirehose_canned_metrics_generated_1.FirehoseMetrics.incomingBytesSum, props);
    }
    metricIncomingRecords(props) {
        return this.cannedMetric(kinesisfirehose_canned_metrics_generated_1.FirehoseMetrics.incomingRecordsSum, props);
    }
    metricBackupToS3Bytes(props) {
        return this.cannedMetric(kinesisfirehose_canned_metrics_generated_1.FirehoseMetrics.backupToS3BytesSum, props);
    }
    metricBackupToS3DataFreshness(props) {
        return this.cannedMetric(kinesisfirehose_canned_metrics_generated_1.FirehoseMetrics.backupToS3DataFreshnessAverage, props);
    }
    metricBackupToS3Records(props) {
        return this.cannedMetric(kinesisfirehose_canned_metrics_generated_1.FirehoseMetrics.backupToS3RecordsSum, props);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ DeliveryStreamName: this.deliveryStreamName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * Options for server-side encryption of a delivery stream.
 */
var StreamEncryption;
(function (StreamEncryption) {
    /**
     * Data in the stream is stored unencrypted.
     */
    StreamEncryption[StreamEncryption["UNENCRYPTED"] = 0] = "UNENCRYPTED";
    /**
     * Data in the stream is stored encrypted by a KMS key managed by the customer.
     */
    StreamEncryption[StreamEncryption["CUSTOMER_MANAGED"] = 1] = "CUSTOMER_MANAGED";
    /**
     * Data in the stream is stored encrypted by a KMS key owned by AWS and managed for use in multiple AWS accounts.
     */
    StreamEncryption[StreamEncryption["AWS_OWNED"] = 2] = "AWS_OWNED";
})(StreamEncryption = exports.StreamEncryption || (exports.StreamEncryption = {}));
/**
 * Create a Kinesis Data Firehose delivery stream
 *
 * @resource AWS::KinesisFirehose::DeliveryStream
 */
class DeliveryStream extends DeliveryStreamBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.deliveryStreamName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_kinesisfirehose_alpha_DeliveryStreamProps(props);
        if (props.destinations.length !== 1) {
            throw new Error(`Only one destination is allowed per delivery stream, given ${props.destinations.length}`);
        }
        const role = (_b = props.role) !== null && _b !== void 0 ? _b : new iam.Role(this, 'Service Role', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
        });
        this.grantPrincipal = role;
        if (props.sourceStream &&
            (props.encryption === StreamEncryption.AWS_OWNED || props.encryption === StreamEncryption.CUSTOMER_MANAGED || props.encryptionKey)) {
            throw new Error('Requested server-side encryption but delivery stream source is a Kinesis data stream. Specify server-side encryption on the data stream instead.');
        }
        if ((props.encryption === StreamEncryption.AWS_OWNED || props.encryption === StreamEncryption.UNENCRYPTED) && props.encryptionKey) {
            throw new Error(`Specified stream encryption as ${StreamEncryption[props.encryption]} but provided a customer-managed key`);
        }
        const encryptionKey = (_c = props.encryptionKey) !== null && _c !== void 0 ? _c : (props.encryption === StreamEncryption.CUSTOMER_MANAGED ? new kms.Key(this, 'Key') : undefined);
        const encryptionConfig = (encryptionKey || (props.encryption === StreamEncryption.AWS_OWNED)) ? {
            keyArn: encryptionKey === null || encryptionKey === void 0 ? void 0 : encryptionKey.keyArn,
            keyType: encryptionKey ? 'CUSTOMER_MANAGED_CMK' : 'AWS_OWNED_CMK',
        } : undefined;
        /*
         * In order for the service role to have access to the encryption key before the delivery stream is created, the
         * CfnDeliveryStream below should have a dependency on the grant returned by the function call below:
         * > `keyGrant?.applyBefore(resource)`
         * However, an error during synthesis is thrown if this is added:
         * > ${Token[PolicyDocument.###]} does not implement DependableTrait
         * Data will not be lost if the permissions are not granted to the service role immediately; Firehose has a 24 hour
         * period where data will be buffered and retried if access is denied to the encryption key. For that reason, it is
         * acceptable to omit the dependency for now. See: https://github.com/aws/aws-cdk/issues/15790
         */
        encryptionKey === null || encryptionKey === void 0 ? void 0 : encryptionKey.grantEncryptDecrypt(role);
        const sourceStreamConfig = props.sourceStream ? {
            kinesisStreamArn: props.sourceStream.streamArn,
            roleArn: role.roleArn,
        } : undefined;
        const readStreamGrant = (_d = props.sourceStream) === null || _d === void 0 ? void 0 : _d.grantRead(role);
        const destinationConfig = props.destinations[0].bind(this, {});
        const resource = new aws_kinesisfirehose_1.CfnDeliveryStream(this, 'Resource', {
            deliveryStreamEncryptionConfigurationInput: encryptionConfig,
            deliveryStreamName: props.deliveryStreamName,
            deliveryStreamType: props.sourceStream ? 'KinesisStreamAsSource' : 'DirectPut',
            kinesisStreamSourceConfiguration: sourceStreamConfig,
            ...destinationConfig,
        });
        (_e = destinationConfig.dependables) === null || _e === void 0 ? void 0 : _e.forEach(dependable => resource.node.addDependency(dependable));
        if (readStreamGrant) {
            resource.node.addDependency(readStreamGrant);
        }
        this.deliveryStreamArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'kinesis',
            resource: 'deliverystream',
            resourceName: this.physicalName,
        });
        this.deliveryStreamName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import an existing delivery stream from its name.
     */
    static fromDeliveryStreamName(scope, id, deliveryStreamName) {
        return this.fromDeliveryStreamAttributes(scope, id, { deliveryStreamName });
    }
    /**
     * Import an existing delivery stream from its ARN.
     */
    static fromDeliveryStreamArn(scope, id, deliveryStreamArn) {
        return this.fromDeliveryStreamAttributes(scope, id, { deliveryStreamArn });
    }
    /**
     * Import an existing delivery stream from its attributes.
     */
    static fromDeliveryStreamAttributes(scope, id, attrs) {
        var _b, _c;
        jsiiDeprecationWarnings._aws_cdk_aws_kinesisfirehose_alpha_DeliveryStreamAttributes(attrs);
        if (!attrs.deliveryStreamName && !attrs.deliveryStreamArn) {
            throw new Error('Either deliveryStreamName or deliveryStreamArn must be provided in DeliveryStreamAttributes');
        }
        const deliveryStreamName = (_b = attrs.deliveryStreamName) !== null && _b !== void 0 ? _b : cdk.Stack.of(scope).splitArn(attrs.deliveryStreamArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!deliveryStreamName) {
            throw new Error(`No delivery stream name found in ARN: '${attrs.deliveryStreamArn}'`);
        }
        const deliveryStreamArn = (_c = attrs.deliveryStreamArn) !== null && _c !== void 0 ? _c : cdk.Stack.of(scope).formatArn({
            service: 'firehose',
            resource: 'deliverystream',
            resourceName: attrs.deliveryStreamName,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
        class Import extends DeliveryStreamBase {
            constructor() {
                var _b;
                super(...arguments);
                this.deliveryStreamName = deliveryStreamName;
                this.deliveryStreamArn = deliveryStreamArn;
                this.grantPrincipal = (_b = attrs.role) !== null && _b !== void 0 ? _b : new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
}
exports.DeliveryStream = DeliveryStream;
_a = JSII_RTTI_SYMBOL_1;
DeliveryStream[_a] = { fqn: "@aws-cdk/aws-kinesisfirehose-alpha.DeliveryStream", version: "2.18.0-alpha.0" };
function setConnections(scope) {
    const stack = cdk.Stack.of(scope);
    const mappingId = '@aws-cdk/aws-kinesisfirehose.CidrBlocks';
    let cfnMapping = constructs_1.Node.of(stack).tryFindChild(mappingId);
    if (!cfnMapping) {
        const mapping = {};
        region_info_1.RegionInfo.regions.forEach((regionInfo) => {
            if (regionInfo.firehoseCidrBlock) {
                mapping[regionInfo.name] = {
                    FirehoseCidrBlock: regionInfo.firehoseCidrBlock,
                };
            }
        });
        cfnMapping = new cdk.CfnMapping(stack, mappingId, {
            mapping,
            lazy: true,
        });
    }
    const cidrBlock = cfnMapping.findInMap(stack.region, 'FirehoseCidrBlock');
    return new ec2.Connections({
        peer: ec2.Peer.ipv4(cidrBlock),
    });
}
//# sourceMappingURL=data:application/json;base64,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