import asyncio
import os
from abc import abstractmethod
from dataclasses import dataclass
from functools import cached_property
from pathlib import Path
from typing import Any, Coroutine, Iterable, List, Optional, Protocol, Sequence, Tuple

from cleo.io.io import IO
from coveo_styles.styles import echo
from coveo_systools.subprocess import DetailedCalledProcessError
from junit_xml import TestCase

from coveo_stew.ci.reporting import generate_report
from coveo_stew.ci.runner_status import RunnerStatus
from coveo_stew.environment import PythonEnvironment
from coveo_stew.stew import PythonProject


class AutoFixRoutineCallable(Protocol):
    def __call__(
        self, environment: PythonEnvironment, **kwargs: Any
    ) -> Coroutine[None, None, None]: ...


class ContinuousIntegrationRunner:
    status: RunnerStatus = RunnerStatus.NotRan
    check_failed_exit_codes: Iterable[int] = []
    outputs_own_report: bool = False  # set to True if the runner produces its own report.

    # implementations may provide an auto fix routine.
    _auto_fix_routine: Optional[AutoFixRoutineCallable] = None

    def __init__(self, io: IO, *, _pyproject: PythonProject) -> None:
        """Implementations may add additional keyword args."""
        self._io = io
        self._pyproject = _pyproject
        self._last_output: List[str] = []
        self._test_cases: List[TestCase] = []
        self._last_exception: Optional[Exception] = None

    @property
    def project(self) -> PythonProject:
        return self._pyproject

    @property
    def io(self) -> IO:
        return self._io

    async def launch(
        self,
        environment: PythonEnvironment = None,
        *extra_args: str,
        auto_fix: bool = False,
    ) -> "ContinuousIntegrationRunner":
        """
        Launch the runner's checks.
        Returns self for convenience with asyncio gather/as_completed/etc.
        """
        self._last_output.clear()
        self._test_cases.clear()
        environment_variables = os.environ.copy()

        # without this, some tools like black will not display emojis correctly
        environment_variables["PYTHONIOENCODING"] = "utf-8"

        # try to set the terminal width if available
        if os.isatty(0):
            try:
                environment_variables["COLUMNS"] = str(os.get_terminal_size().columns)
            except OSError:
                pass

        try:
            self.status = await self._launch(environment, *extra_args, env=environment_variables)
        except DetailedCalledProcessError as exception:
            if exception.returncode in self.check_failed_exit_codes:
                self.status = RunnerStatus.CheckFailed
            else:
                self.status = RunnerStatus.Error
                self._last_exception = exception
            self._last_output.extend(exception.decode_output().split("\n"))
            self._last_output.extend(exception.decode_stderr().split("\n"))

        if all((auto_fix, self.supports_auto_fix, self.status == RunnerStatus.CheckFailed)):
            echo.noise("Errors founds; launching auto-fix routine.")
            assert self._auto_fix_routine is not None  # mypy
            await self._auto_fix_routine(environment, env=environment_variables)

            # it should pass now!
            await self.launch(environment, *extra_args)
            if self.status == RunnerStatus.CheckFailed:
                echo.error("The auto fix routine was launched but the check is still failing.")
            else:
                echo.success("Auto fix was a success. Good job soldier!")

        if not self.outputs_own_report:
            self._output_generic_report(environment)

        return self

    @property
    @abstractmethod
    def name(self) -> str:
        """The friendly name of this runner."""

    @property
    def supports_auto_fix(self) -> bool:
        """Does this runner support autofix?"""
        return self._auto_fix_routine is not None

    @abstractmethod
    async def _launch(
        self, environment: PythonEnvironment, *extra_args: str, **kwargs: Any
    ) -> RunnerStatus:
        """Launch the continuous integration check using the given environment and store the output."""

    def has_output(self) -> bool:
        return bool(self._last_output)

    def echo_output(self) -> None:
        """Echo the output of the run(s) to the user."""
        if not self.has_output():
            return
        echo.noise(self.last_output(), pad_after=True)

    def last_output(self) -> str:
        return "\n".join(self._last_output).strip()

    @property
    def last_exception(self) -> Optional[Exception]:
        return self._last_exception

    def report_path(self, environment: PythonEnvironment) -> Path:
        """The report path for the current invocation. e.g.: ci.py3.6.2.mypy.coveo-functools.xml"""
        report_folder = self._pyproject.project_path / ".ci"
        if not report_folder.exists():
            report_folder.mkdir()

        return report_folder / ".".join(
            (
                "ci",
                environment.pretty_python_version,
                self.name,
                self._pyproject.poetry.package.pretty_name,
                "xml",
            )
        )

    def _output_generic_report(self, environment: PythonEnvironment) -> None:
        test_case = TestCase(
            self.name, classname=f"ci.{self._pyproject.poetry.package.pretty_name}"
        )
        if self.status is RunnerStatus.Error:
            test_case.add_error_info(
                "An error occurred, the test was unable to complete.",
                self.last_output(),
            )
        elif self.status is RunnerStatus.CheckFailed:
            test_case.add_failure_info("The test completed; errors were found.", self.last_output())
        generate_report(
            self._pyproject.poetry.package.pretty_name, self.report_path(environment), [test_case]
        )

    def store_output(self, output: Optional[str]) -> None:
        if output:
            self._last_output.extend(output.strip().splitlines())

    def __str__(self) -> str:
        return self.name


@dataclass
class CIPlan:
    """
    The CIPlan holds all the runners for an environment
    and orchestrates the workflow of launching them.
    """

    environment: PythonEnvironment
    checks: Sequence[ContinuousIntegrationRunner]
    parallel: bool

    @cached_property
    def autofix_checks(self) -> List[ContinuousIntegrationRunner]:
        """
        Autofix checks receive a special treatment since they will potentially change line numbers,
        making reports inaccurate.
        """
        return [check for check in self.checks if check.supports_auto_fix]

    @cached_property
    def non_autofix_checks(self) -> List[ContinuousIntegrationRunner]:
        return [check for check in self.checks if check not in self.autofix_checks]

    async def orchestrate(self, auto_fix: bool = False, show_success_output: bool = False) -> None:
        """Orchestrates this CIPlan by launching runners in the correct order."""
        runs: List[Run] = []
        echo.step(
            f"Planned {len(self.checks)} runners for {self.environment.pretty_python_version}"
        )

        if auto_fix:
            # run autofix first, in parallel
            runs.append(run := Run(self.environment, self.autofix_checks))
            await run.run_and_report(
                parallel=self.parallel, show_success_output=show_success_output
            )

            if run.overall_status is RunnerStatus.CheckFailed:
                await run.run_and_report(auto_fix=True, feedback=False, parallel=False)
                await run.run_and_report(
                    parallel=self.parallel, show_success_output=show_success_output
                )  # verify that autofix worked

            # run all other runners
            runs.append(run := Run(self.environment, self.non_autofix_checks))
            await run.run_and_report(
                parallel=self.parallel, show_success_output=show_success_output
            )

        else:
            runs.append(run := Run(self.environment, self.checks))
            await run.run_and_report(
                parallel=self.parallel, show_success_output=show_success_output
            )

        overall_status = get_overall_run_status(*runs)

        status_to_style_map = {
            RunnerStatus.Success: echo.success,
            RunnerStatus.CheckFailed: echo.warning,
            RunnerStatus.Error: echo.error,
            RunnerStatus.NotRan: echo.outcome,
        }

        status_to_style_map[overall_status](
            f"The CI run for {self.environment.pretty_python_version} completed with status: {overall_status}"
        )


@dataclass
class Run:
    """The Run is a stateful object that runs checks and reports the results to the user."""

    environment: PythonEnvironment
    checks: Sequence[ContinuousIntegrationRunner]

    @cached_property
    def exceptions(
        self,
    ) -> List[Tuple[Optional[ContinuousIntegrationRunner], Exception]]:
        """Exceptions are stored here after the run. Exceptions are cleared when `run_and_report` is called."""
        return []

    @property
    def overall_status(self) -> RunnerStatus:
        return get_overall_run_status(self)

    async def run_and_report(
        self,
        auto_fix: bool = False,
        feedback: bool = True,
        parallel: bool = True,
        show_success_output: bool = False,
    ) -> None:
        """Launch the runners and report the results to the user."""
        self.exceptions.clear()

        if auto_fix and parallel:
            raise AssertionError(
                "Some dev made a mistake; parallel and autofix are mutually exclusive!"
            )

        if parallel:
            for next_result in asyncio.as_completed(
                [runner.launch(self.environment, auto_fix=False) for runner in self.checks]
            ):
                try:
                    result = await next_result
                except Exception as exc:
                    self.exceptions.append((None, exc))
                    continue

                self._report(result, feedback=feedback, show_success_output=show_success_output)

        else:
            for runner in self.checks:
                self._report(
                    await runner.launch(self.environment, auto_fix=auto_fix),
                    feedback=feedback,
                )

        if self.exceptions:
            for check, exception in self.exceptions:
                if check is None:
                    echo.error("A runner created an exception: ", pad_before=True)
                    echo.noise(exception, pad_after=True)
                else:
                    echo.error(f"The runner {check} created an exception: ", pad_before=True)
                    echo.noise(exception, pad_after=True)

            echo.error(
                "One or more checks were not able to complete:",
                pad_before=True,
                pad_after=False,
                emoji="robot",
            )
            echo.warning(
                "To have stew treat an exit code as a check failure instead of an error, use `check-failed-exit-codes`",
                item=True,
                pad_before=False,
                pad_after=False,
            )
            echo.warning(
                "https://github.com/coveo/stew/blob/main/README.md#options",
                item=True,
                pad_before=False,
                pad_after=False,
            )
            echo.warning(
                "Use the working directory and command (printed above) to invoke the command from the shell manually",
                item=True,
                pad_before=False,
                pad_after=True,
            )

    def _report(
        self,
        check: ContinuousIntegrationRunner,
        feedback: bool = True,
        show_success_output: bool = False,
    ) -> None:
        """Reports on a completed check."""
        if check.status is RunnerStatus.Error:
            self.exceptions.append((check, check.last_exception))

        if feedback:
            if check.status is RunnerStatus.Success:
                echo.normal(f" PASSED: {check}", emoji="heavy_check_mark", fg="green")
                if show_success_output or check.project.verbose:
                    if not check.has_output():
                        echo.noise(f"({check.name} produced no output)", pad_after=True)
                    else:
                        check.echo_output()

            elif check.status is RunnerStatus.CheckFailed:
                echo.warning(
                    f"{check} [{check.project.poetry.package.pretty_name}] reported issues:",
                    pad_before=True,
                    pad_after=False,
                )
                check.echo_output()

            elif check.status is RunnerStatus.Error:
                echo.error(
                    f"The ci runner {check} failed to complete "
                    f"due to an environment or configuration error."
                )
                check.echo_output()


def get_overall_run_status(*runs: Run) -> RunnerStatus:
    """Return the overall run status for the provided runs."""
    statuses = [check.status for run in runs for check in run.checks]
    for status in RunnerStatus.Error, RunnerStatus.CheckFailed, RunnerStatus.Success:
        if status in statuses:
            return status

    return RunnerStatus.NotRan
