"""
Projects marked as pydev behave as "one ring to rule them all"; its `pyproject.toml` file links most/all of the
python projects in a repository for developer convenience.

It has some additional features and tricks that do not apply to other projects. This is centralized here.
"""

from typing import Any, Dict, Generator, Tuple

import tomlkit
from cleo.io.io import IO
from coveo_styles.styles import ExitWithFailure
from coveo_systools.filesystem import safe_text_write
from poetry.core.packages.dependency import Dependency
from poetry.core.packages.path_dependency import PathDependency
from poetry.poetry import Poetry
from tomlkit.items import Item as TomlItem
from tomlkit.items import Table
from tomlkit.items import item as toml_item

from coveo_stew.exceptions import NotAPoetryProject, StewException
from coveo_stew.stew import PythonProject


class NotPyDevProject(StewException): ...


def is_pydev_project(poetry: Poetry) -> bool:
    """Returns true when a project is a pydev project. Typically used as a predicate for `discover_pyprojects`."""
    try:
        return poetry.pyproject.data["tool"]["stew"]["pydev"] is True  # type: ignore[index]
    except KeyError:
        return False
    except TypeError as ex:
        raise ExitWithFailure(
            failures=[
                poetry.pyproject_path,
                "Error while checking if the project is a pydev project.",
                "The [tool.stew] section of the `pyproject.toml` file is malformed.",
            ],
            suggestions=[
                f"Ensure the {poetry.pyproject_path} file has a valid [tool.stew] section.",
                "For instance, this specifies a pydev project:\n\n[tool.stew]\npydev = true\n\n",
            ],
        ) from ex


def pull_and_write_dev_requirements(
    io: IO, project: PythonProject, *, dry_run: bool = False
) -> bool:
    """Pulls the dev requirement from dependencies into pydev's dev requirements."""
    if not project.options.pydev:
        raise NotPyDevProject(f"{project.project_path}: Not a PyDev project.")

    # prepare a toml container for our data
    toml: Dict[str, Any] = tomlkit.loads(project.poetry.pyproject_path.read_text())

    all_dev_dependencies: Table = tomlkit.table()

    # extract the dev requirements from the local dependencies
    for item in sorted(_dev_dependencies_of_dependencies(io, project)):
        all_dev_dependencies.add(*item)

    # the py dev environment package has no code, no tests, no entrypoints, no nothin'!
    # as such, dev dependencies are irrelevant; we reserve the section for the current purpose.
    all_dev_dependencies.comment("this dependency group is autogenerated; do not edit manually!")

    if "dev-dependencies" in toml["tool"]["poetry"]:
        # remove the dev dependencies section if it exists, as we are replacing it with a group
        # this is to avoid confusion with the dev dependencies of the project itself
        io.write_line(
            "<fg=yellow>Removing the deprecated dev-dependencies section from pyproject.toml</>"
        )
        toml["tool"]["poetry"].pop("dev-dependencies", None)

    groups = toml["tool"]["poetry"]["group"]
    if "dev" in groups and "stew-pydev" not in groups:
        # assume that the dev section is ours; remove it and replace it with the stew-pydev group.
        io.write_line(
            "<fg=yellow>Removing the dev group from pyproject.toml. It has been replaced by `stew-pydev`.</>"
        )
        groups.pop("dev", None)

    toml["tool"]["poetry"]["group"] = {"stew-pydev": {"dependencies": all_dev_dependencies}}

    if safe_text_write(
        project.poetry.pyproject_path,
        "\n".join(_format_toml(tomlkit.dumps(toml))),
        only_if_changed=True,
        dry_run=dry_run,
    ):
        if not dry_run and project.lock_if_needed():
            project.install()
        return True
    return False


def _format_toml(toml_content: str) -> Generator[str, None, None]:
    """tomlkit sometimes forgets to add empty lines before sections."""
    first = True
    for line in toml_content.split("\n"):
        if not first and line.strip().startswith("["):
            yield "\n"  # extra empty line between each section means 2 empty lines between sections
        if line.strip():
            yield line
        first = False


def _dev_dependencies_of_dependencies(
    io: IO,
    project: PythonProject,
) -> Generator[Tuple[str, TomlItem], None, None]:
    """Yields the dev dependencies of this project's dependencies."""
    direct_dependencies: dict[str, Dependency] = {d.pretty_name: d for d in project.dependencies}
    dependencies: dict[str, Dependency] = {}
    for local_dependency in direct_dependencies.values():
        if not isinstance(local_dependency, PathDependency):
            continue  # we only pull the dev dependencies of local path dependencies
        assert not local_dependency.path.is_absolute()

        try:
            local_project = PythonProject(
                io, project.project_path / local_dependency.path, verbose=project.verbose
            )
        except NotAPoetryProject as exception:
            raise ExitWithFailure(
                suggestions=f"Add a `pyproject.toml` file in {local_dependency.path}",
                failures="Local dependencies must also be poetry projects.",
            ) from exception

        for dev_dependency in local_project.dev_dependencies:
            if dev_dependency.pretty_name in direct_dependencies:
                # if the dependency is already a direct dependency, we do not need to add it again
                continue
            if dev_dependency.pretty_name not in dependencies:
                dependencies[dev_dependency.pretty_name] = dev_dependency
            else:
                # tighten the constraint to satisfy what we have seen so far
                seen_dependency = dependencies[dev_dependency.pretty_name]
                new_constraint = dev_dependency.constraint.intersect(seen_dependency.constraint)
                dependencies[dev_dependency.pretty_name] = seen_dependency.with_constraint(
                    new_constraint
                )

    for name, dependency in dependencies.items():
        if isinstance(dependency, PathDependency):
            value: Any = tomlkit.inline_table()
            value.append(
                "path",
                str(dependency.full_path.relative_to(project.project_path)),
            )
        else:
            value = dependency.pretty_constraint
        yield dependency.pretty_name, toml_item(value)
