-- Setup companies and contacts tables with RLS policies
-- This migration handles the core CRM data structure

-- Create companies table
CREATE TABLE IF NOT EXISTS public.companies (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  industry TEXT,
  website TEXT,
  size TEXT,
  location TEXT,
  notes TEXT,
  logo_url TEXT,
  created_by UUID REFERENCES auth.users(id) NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now() NOT NULL,
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT now() NOT NULL
);

-- Enable RLS on companies
ALTER TABLE public.companies ENABLE ROW LEVEL SECURITY;

-- Create contacts table
CREATE TABLE IF NOT EXISTS public.contacts (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  email TEXT,
  phone TEXT,
  company_id UUID REFERENCES public.companies(id) ON DELETE SET NULL,
  position TEXT,
  status TEXT DEFAULT 'active',
  notes TEXT,
  avatar_url TEXT,
  tags TEXT[],
  created_by UUID REFERENCES auth.users(id) NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now() NOT NULL,
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT now() NOT NULL
);

-- Enable RLS on contacts
ALTER TABLE public.contacts ENABLE ROW LEVEL SECURITY;

-- Create RLS policies for companies

-- Team members can read all companies
CREATE POLICY "Team members can read all companies" ON public.companies
  FOR SELECT
  USING (true);

-- Only managers and admins can create companies
CREATE POLICY "Managers and admins can create companies" ON public.companies
  FOR INSERT
  WITH CHECK ((SELECT role IN ('admin', 'manager') FROM public.profiles WHERE id = auth.uid()));

-- Managers and admins can update companies
CREATE POLICY "Managers and admins can update companies" ON public.companies
  FOR UPDATE
  USING ((SELECT role IN ('admin', 'manager') FROM public.profiles WHERE id = auth.uid()));

-- Only admins can delete companies
CREATE POLICY "Only admins can delete companies" ON public.companies
  FOR DELETE
  USING ((SELECT role = 'admin' FROM public.profiles WHERE id = auth.uid()));

-- Create RLS policies for contacts

-- Team members can read all contacts
CREATE POLICY "Team members can read all contacts" ON public.contacts
  FOR SELECT
  USING (true);

-- Users can create contacts
CREATE POLICY "Users can create contacts" ON public.contacts
  FOR INSERT
  WITH CHECK (auth.uid() IS NOT NULL);

-- Users can update contacts they created and managers/admins can update any
CREATE POLICY "Users can update own contacts" ON public.contacts
  FOR UPDATE
  USING (created_by = auth.uid() OR 
        (SELECT role IN ('admin', 'manager') FROM public.profiles WHERE id = auth.uid()));

-- Only admins and managers can delete contacts
CREATE POLICY "Only admins and managers can delete contacts" ON public.contacts
  FOR DELETE
  USING ((SELECT role IN ('admin', 'manager') FROM public.profiles WHERE id = auth.uid()));

-- Add triggers for updating timestamps
CREATE TRIGGER update_companies_timestamp
BEFORE UPDATE ON public.companies
FOR EACH ROW
EXECUTE PROCEDURE update_modified_column();

CREATE TRIGGER update_contacts_timestamp
BEFORE UPDATE ON public.contacts
FOR EACH ROW
EXECUTE PROCEDURE update_modified_column();

-- Insert sample companies data
-- We'll use a function to ensure we have a valid user ID to reference
DO $$
DECLARE
  admin_user_id UUID;
BEGIN
  -- Get the first user ID (assuming at least one user exists)
  SELECT id INTO admin_user_id FROM auth.users LIMIT 1;
  
  -- Fall back to a static UUID if no users exist yet
  IF admin_user_id IS NULL THEN
    admin_user_id := '00000000-0000-0000-0000-000000000000';
  END IF;
  
  -- Insert sample companies
  INSERT INTO public.companies (name, industry, website, size, location, notes, logo_url, created_by)
  VALUES
    ('Acme Corporation', 'Technology', 'https://acme.example.com', '50-100', 'San Francisco, CA', 'Leading provider of innovative solutions', 'https://ui-avatars.com/api/?name=Acme&background=0D8ABC&color=fff', admin_user_id),
    ('TechFlow Inc.', 'Software', 'https://techflow.example.com', '10-50', 'Austin, TX', 'B2B SaaS platform', 'https://ui-avatars.com/api/?name=Tech&background=FF5733&color=fff', admin_user_id),
    ('Globex Industries', 'Manufacturing', 'https://globex.example.com', '500+', 'Chicago, IL', 'Industrial equipment manufacturer', 'https://ui-avatars.com/api/?name=Globex&background=27AE60&color=fff', admin_user_id),
    ('Atlas Consulting', 'Consulting', 'https://atlas.example.com', '10-50', 'New York, NY', 'Business strategy consultancy', 'https://ui-avatars.com/api/?name=Atlas&background=8E44AD&color=fff', admin_user_id),
    ('Oceanic Airlines', 'Travel', 'https://oceanic.example.com', '100-500', 'Los Angeles, CA', 'Regional airline', 'https://ui-avatars.com/api/?name=Oceanic&background=2980B9&color=fff', admin_user_id);
END $$;

-- Insert sample contacts data
DO $$
DECLARE
  admin_user_id UUID;
  company_ids UUID[];
  company_id UUID;
BEGIN
  -- Get the first user ID
  SELECT id INTO admin_user_id FROM auth.users LIMIT 1;
  
  -- Fall back to a static UUID if no users exist yet
  IF admin_user_id IS NULL THEN
    admin_user_id := '00000000-0000-0000-0000-000000000000';
  END IF;
  
  -- Get all company IDs
  SELECT array_agg(id) INTO company_ids FROM public.companies;
  
  -- If no companies exist, exit
  IF company_ids IS NULL OR array_length(company_ids, 1) = 0 THEN
    RAISE NOTICE 'No companies found, cannot create contacts';
    RETURN;
  END IF;
  
  -- Insert contacts for each company
  FOREACH company_id IN ARRAY company_ids
  LOOP
    -- Insert 3-5 contacts per company
    INSERT INTO public.contacts (name, email, phone, company_id, position, status, notes, avatar_url, tags, created_by)
    VALUES
      ('John Doe', 'john.doe@example.com', '+1 (555) 123-4567', company_id, 'CEO', 'active', 'Key decision maker', 'https://randomuser.me/api/portraits/men/1.jpg', ARRAY['vip', 'decision-maker'], admin_user_id),
      ('Jane Smith', 'jane.smith@example.com', '+1 (555) 987-6543', company_id, 'CTO', 'active', 'Technical contact', 'https://randomuser.me/api/portraits/women/2.jpg', ARRAY['technical'], admin_user_id),
      ('Michael Johnson', 'michael.j@example.com', '+1 (555) 456-7890', company_id, 'Sales Director', 'active', 'Sales contact', 'https://randomuser.me/api/portraits/men/3.jpg', ARRAY['sales'], admin_user_id);
  END LOOP;
END $$; 