"use client"

import { useState } from "react"
import { Building2, Globe, MapPin, Search, Users } from "lucide-react"
import Image from "next/image"
import Link from "next/link"

import { companies } from "@/lib/mock-data"
import { Card, CardContent } from "@/components/ui/card"
import { Input } from "@/components/ui/input"

export default function CompaniesPage() {
  const [searchTerm, setSearchTerm] = useState("")

  // Filter companies based on search term
  const filteredCompanies = companies.filter(company => 
    company.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    (company.industry && company.industry.toLowerCase().includes(searchTerm.toLowerCase())) ||
    (company.location && company.location.toLowerCase().includes(searchTerm.toLowerCase()))
  )

  return (
    <div className="p-6 space-y-8">
      <div className="space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Companies</h2>
        <p className="text-muted-foreground">
          Manage your client and prospect companies.
        </p>
      </div>

      <div className="flex justify-between items-center">
        <div className="relative w-full max-w-sm">
          <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
          <Input
            type="text"
            placeholder="Search companies..."
            className="pl-9"
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
          />
        </div>
        <button className="bg-primary text-primary-foreground hover:bg-primary/90 px-4 py-2 rounded-md text-sm font-medium">
          Add Company
        </button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {filteredCompanies.map((company) => (
          <Link key={company.id} href={`/companies/${company.id}`}>
            <Card className="cursor-pointer h-full hover:shadow-md transition-shadow">
              <CardContent className="pt-6">
                <div className="flex items-start space-x-4">
                  {company.logo ? (
                    <div className="h-16 w-16 rounded-md overflow-hidden relative flex-shrink-0">
                      <Image
                        src={company.logo}
                        alt={company.name}
                        width={64}
                        height={64}
                        className="object-cover"
                      />
                    </div>
                  ) : (
                    <div className="h-16 w-16 rounded-md bg-muted flex items-center justify-center flex-shrink-0">
                      <Building2 className="h-8 w-8 text-muted-foreground" />
                    </div>
                  )}
                  <div className="flex-1 space-y-1 min-w-0">
                    <h3 className="font-semibold text-lg truncate">{company.name}</h3>
                    {company.industry && (
                      <p className="text-sm text-muted-foreground">{company.industry}</p>
                    )}
                    <div className="flex flex-wrap gap-3 mt-3">
                      {company.location && (
                        <div className="flex items-center text-xs text-muted-foreground">
                          <MapPin className="h-3 w-3 mr-1" />
                          <span>{company.location}</span>
                        </div>
                      )}
                      {company.website && (
                        <div className="flex items-center text-xs text-muted-foreground">
                          <Globe className="h-3 w-3 mr-1" />
                          <span className="truncate max-w-[150px]">
                            {company.website.replace(/^https?:\/\//, '')}
                          </span>
                        </div>
                      )}
                      <div className="flex items-center text-xs text-muted-foreground">
                        <Users className="h-3 w-3 mr-1" />
                        <span>{company.size || 'Unknown'}</span>
                      </div>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </Link>
        ))}
      </div>
    </div>
  )
} 