"use client"

import { useState, useEffect } from "react"
import { Search, Plus } from "lucide-react"
import { getContacts } from "@/lib/data-service"
import { Contact } from "@/types/crm"
import { formatDistanceToNow } from "date-fns"

export default function ContactsPage() {
  const [contacts, setContacts] = useState<Contact[]>([])
  const [searchTerm, setSearchTerm] = useState("")
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    async function loadContacts() {
      setIsLoading(true)
      try {
        const contactsData = await getContacts()
        setContacts(contactsData)
      } catch (error) {
        console.error("Error loading contacts:", error)
      } finally {
        setIsLoading(false)
      }
    }

    loadContacts()
  }, [])

  // Filter contacts based on search term
  const filteredContacts = contacts.filter(contact => 
    contact.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    (contact.email && contact.email.toLowerCase().includes(searchTerm.toLowerCase())) ||
    (contact.company && contact.company.toLowerCase().includes(searchTerm.toLowerCase()))
  )

  // Helper to format date
  const formatDate = (dateString?: string) => {
    if (!dateString) return "Never"
    try {
      return formatDistanceToNow(new Date(dateString), { addSuffix: true })
    } catch (e) {
      return dateString
    }
  }

  return (
    <div className="p-6 space-y-8">
      <div className="space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Contacts</h2>
        <p className="text-muted-foreground">
          Manage your contacts and keep track of your customer relationships.
        </p>
      </div>

      <div className="flex justify-between items-center">
        <div className="relative w-full max-w-sm">
          <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
          <input
            type="text"
            placeholder="Search contacts..."
            className="w-full rounded-md border border-input px-9 py-2 text-sm"
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
          />
        </div>
        <button className="bg-primary text-primary-foreground hover:bg-primary/90 px-4 py-2 rounded-md text-sm font-medium flex items-center">
          <Plus className="h-4 w-4 mr-2" />
          Add Contact
        </button>
      </div>

      {isLoading ? (
        <div className="p-12 flex justify-center">
          <div className="animate-pulse text-lg">Loading contacts...</div>
        </div>
      ) : (
        <div className="border rounded-md">
          <div className="grid grid-cols-6 bg-muted/50 p-3 text-sm font-medium text-muted-foreground">
            <div className="col-span-2">Name/Email</div>
            <div>Company</div>
            <div>Phone</div>
            <div>Last Activity</div>
            <div>Status</div>
          </div>
          {filteredContacts.length === 0 ? (
            <div className="p-8 text-center text-muted-foreground">
              {searchTerm ? "No contacts found matching your search" : "No contacts found. Add your first contact to get started!"}
            </div>
          ) : (
            <div className="divide-y">
              {filteredContacts.map(contact => (
                <div key={contact.id} className="grid grid-cols-6 p-3 text-sm items-center hover:bg-muted/30">
                  <div className="col-span-2">
                    <div className="font-medium">{contact.name}</div>
                    <div className="text-muted-foreground">{contact.email}</div>
                  </div>
                  <div>{contact.company || '-'}</div>
                  <div>{contact.phone || '-'}</div>
                  <div>{formatDate(contact.createdAt)}</div>
                  <div>
                    <span className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium ${
                      contact.status === 'active' 
                        ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300'
                        : 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-300'
                    }`}>
                      {contact.status ? contact.status.charAt(0).toUpperCase() + contact.status.slice(1) : 'Unknown'}
                    </span>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      )}
    </div>
  )
} 