"use client"

import { useState, useEffect } from "react"
import Link from "next/link"
import { 
  ArrowUp, 
  ArrowDown, 
  CheckCircle2, 
  Clock, 
  CreditCard, 
  DollarSign, 
  Globe, 
  BarChart3,
  Users, 
  Phone,
  Mail,
  Activity,
  Calendar
} from "lucide-react"

import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from "@/components/ui/button"
import { useAuth } from "@/components/auth/AuthProvider"
import { ActivityItem } from "@/components/dashboard/ActivityItem"

// Replace mock data imports with data service
import { 
  getActivities,
  getDeals, 
  getContacts, 
  getTasks, 
  getCompanies
} from "@/lib/data-service"
import { Contact, Deal, Task, ActivityLog, Company } from "@/types/crm"

export default function DashboardPage() {
  const { user } = useAuth()
  
  // Add states for database-fetched data
  const [deals, setDeals] = useState<Deal[]>([])
  const [contacts, setContacts] = useState<Contact[]>([])
  const [companies, setCompanies] = useState<Company[]>([])
  const [tasks, setTasks] = useState<Task[]>([])
  const [activities, setActivities] = useState<ActivityLog[]>([])
  const [isLoading, setIsLoading] = useState(true)
  
  // Fetch data from Supabase on component mount
  useEffect(() => {
    async function loadData() {
      setIsLoading(true)
      try {
        const [
          dealsData, 
          contactsData, 
          companiesData, 
          tasksData, 
          activitiesData
        ] = await Promise.all([
          getDeals(),
          getContacts(),
          getCompanies(),
          getTasks(),
          getActivities(10)
        ])
        
        setDeals(dealsData)
        setContacts(contactsData)
        setCompanies(companiesData)
        setTasks(tasksData)
        setActivities(activitiesData)
      } catch (error) {
        console.error("Error loading dashboard data:", error)
      } finally {
        setIsLoading(false)
      }
    }
    
    loadData()
  }, [])
  
  // Calculate dashboard metrics
  const totalDealsValue = deals.reduce((total, deal) => total + (deal.value || 0), 0)
  const openDealsCount = deals.filter(deal => deal.stage !== 'closed').length
  const openDealsValue = deals
    .filter(deal => deal.stage !== 'closed')
    .reduce((total, deal) => total + (deal.value || 0), 0)
  
  const pendingTasksCount = tasks.filter(task => task.status === 'pending' || task.status === 'in_progress').length
  const overdueTasksCount = tasks.filter(task => {
    if (!task.dueDate || task.status === 'completed') return false
    return new Date(task.dueDate) < new Date() && task.status !== 'completed'
  }).length
  
  return (
    <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <h2 className="text-3xl font-bold tracking-tight">Dashboard</h2>
        <div className="flex items-center gap-2">
          <Button variant="outline" size="sm" className="h-8 gap-1">
            <Calendar className="h-3.5 w-3.5" />
            <span className="hidden sm:inline">Today</span>
          </Button>
          <Button size="sm" className="h-8 gap-1">
            <Activity className="h-3.5 w-3.5" />
            <span className="hidden sm:inline">Activity</span>
          </Button>
        </div>
      </div>
      
      {isLoading ? (
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          {[...Array(4)].map((_, i) => (
            <Card key={i} className="animate-pulse">
              <CardHeader className="pb-2">
                <div className="h-4 bg-muted rounded w-24"></div>
              </CardHeader>
              <CardContent>
                <div className="h-7 bg-muted rounded w-16"></div>
                <div className="h-4 bg-muted/50 rounded w-32 mt-2"></div>
              </CardContent>
            </Card>
          ))}
        </div>
      ) : (
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2 space-y-0">
              <CardTitle className="text-sm font-medium">Total Deals Value</CardTitle>
              <DollarSign className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">${totalDealsValue.toLocaleString()}</div>
              <p className="text-xs text-muted-foreground">
                {openDealsCount} open deals worth ${openDealsValue.toLocaleString()}
              </p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2 space-y-0">
              <CardTitle className="text-sm font-medium">Contacts</CardTitle>
              <Users className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{contacts.length}</div>
              <p className="text-xs text-muted-foreground">
                From {companies.length} companies
              </p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2 space-y-0">
              <CardTitle className="text-sm font-medium">Tasks</CardTitle>
              <CheckCircle2 className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{pendingTasksCount}</div>
              <p className="text-xs text-muted-foreground">
                {overdueTasksCount} overdue
              </p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2 space-y-0">
              <CardTitle className="text-sm font-medium">Companies</CardTitle>
              <Globe className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{companies.length}</div>
              <p className="text-xs text-muted-foreground">
                Across {[...new Set(companies.map(c => c.industry))].filter(Boolean).length} industries
              </p>
            </CardContent>
          </Card>
        </div>
      )}
    
      <Tabs defaultValue="overview">
        <TabsList>
          <TabsTrigger value="overview">Overview</TabsTrigger>
          <TabsTrigger value="analytics" disabled>Analytics</TabsTrigger>
          <TabsTrigger value="reports" disabled>Reports</TabsTrigger>
          <TabsTrigger value="notifications" disabled>Notifications</TabsTrigger>
        </TabsList>
        <TabsContent value="overview" className="space-y-4">
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-7">
            <Card className="col-span-4">
              <CardHeader>
                <CardTitle>Recent Activity</CardTitle>
                <CardDescription>
                  Your team's activity across deals, contacts, and tasks
                </CardDescription>
              </CardHeader>
              <CardContent>
                {isLoading ? (
                  <div className="space-y-4">
                    {[...Array(5)].map((_, i) => (
                      <div key={i} className="flex gap-4 animate-pulse">
                        <div className="h-9 w-9 rounded-full bg-muted"></div>
                        <div className="space-y-2 flex-1">
                          <div className="h-4 bg-muted rounded w-3/4"></div>
                          <div className="h-3 bg-muted/50 rounded w-1/2"></div>
                        </div>
                      </div>
                    ))}
                  </div>
                ) : activities.length === 0 ? (
                  <div className="text-center py-6 text-muted-foreground">
                    <Activity className="h-8 w-8 mx-auto mb-2 opacity-50" />
                    <p>No activities recorded yet</p>
                    <p className="text-sm">Activities will appear here as you and your team work</p>
                  </div>
                ) : (
                  <div className="space-y-4">
                    {activities.map((activity) => (
                      <ActivityItem key={activity.id} activity={activity} />
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
            <Card className="col-span-3">
              <CardHeader>
                <CardTitle>Upcoming Tasks</CardTitle>
                <CardDescription>
                  Tasks due in the next 7 days
                </CardDescription>
              </CardHeader>
              <CardContent>
                {isLoading ? (
                  <div className="space-y-4">
                    {[...Array(3)].map((_, i) => (
                      <div key={i} className="animate-pulse">
                        <div className="h-4 bg-muted rounded w-2/3 mb-2"></div>
                        <div className="h-3 bg-muted/50 rounded w-1/2"></div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="space-y-4">
                    {tasks
                      .filter(task => task.status !== 'completed' && task.dueDate)
                      .sort((a, b) => new Date(a.dueDate!).getTime() - new Date(b.dueDate!).getTime())
                      .slice(0, 5)
                      .map(task => (
                        <div key={task.id} className="flex items-start gap-2">
                          <div className={`p-1.5 rounded-full ${
                            task.priority === 'high' ? 'bg-destructive/20 text-destructive' :
                            task.priority === 'medium' ? 'bg-yellow-100 text-yellow-700 dark:bg-yellow-700/20 dark:text-yellow-500' :
                            'bg-muted text-muted-foreground'
                          }`}>
                            <Clock className="h-3.5 w-3.5" />
                          </div>
                          <div className="space-y-0.5">
                            <p className="text-sm font-medium">{task.title}</p>
                            <div className="flex gap-2 text-xs text-muted-foreground">
                              <span>{new Date(task.dueDate!).toLocaleDateString()}</span>
                              {task.type === 'call' && <Phone className="h-3 w-3" />}
                              {task.type === 'email' && <Mail className="h-3 w-3" />}
                            </div>
                          </div>
                        </div>
                      ))
                    }
                    {tasks.filter(task => task.status !== 'completed' && task.dueDate).length === 0 && (
                      <div className="text-center py-4 text-muted-foreground">
                        <CheckCircle2 className="h-8 w-8 mx-auto mb-2 opacity-50" />
                        <p>No upcoming tasks</p>
                        <Button variant="link" size="sm" className="mt-2" asChild>
                          <Link href="/tasks">Create a task</Link>
                        </Button>
                      </div>
                    )}
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
            <Card>
              <CardHeader>
                <CardTitle>Latest Deals</CardTitle>
                <CardDescription>
                  Recent deals in your pipeline
                </CardDescription>
              </CardHeader>
              <CardContent>
                {isLoading ? (
                  <div className="space-y-4">
                    {[...Array(3)].map((_, i) => (
                      <div key={i} className="animate-pulse">
                        <div className="h-4 bg-muted rounded w-2/3 mb-2"></div>
                        <div className="h-3 bg-muted/50 rounded w-1/2"></div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="space-y-4">
                    {deals.slice(0, 5).map(deal => (
                      <div key={deal.id} className="flex justify-between items-start gap-4">
                        <div className="space-y-0.5">
                          <p className="text-sm font-medium">{deal.title}</p>
                          <div className="flex items-center gap-1">
                            <div className={`h-2 w-2 rounded-full ${deal.stageColor || 'bg-blue-500'}`}></div>
                            <p className="text-xs text-muted-foreground">{deal.stageTitle}</p>
                          </div>
                        </div>
                        <p className="text-sm font-medium">${deal.value?.toLocaleString()}</p>
                      </div>
                    ))}
                    {deals.length === 0 && (
                      <div className="text-center py-4 text-muted-foreground">
                        <CreditCard className="h-8 w-8 mx-auto mb-2 opacity-50" />
                        <p>No deals created yet</p>
                        <Button variant="link" size="sm" className="mt-2" asChild>
                          <Link href="/deals">Create a deal</Link>
                        </Button>
                      </div>
                    )}
                  </div>
                )}
              </CardContent>
            </Card>
            <Card>
              <CardHeader>
                <CardTitle>Top Contacts</CardTitle>
                <CardDescription>
                  Your most engaged contacts
                </CardDescription>
              </CardHeader>
              <CardContent>
                {isLoading ? (
                  <div className="space-y-4">
                    {[...Array(3)].map((_, i) => (
                      <div key={i} className="animate-pulse">
                        <div className="h-4 bg-muted rounded w-2/3 mb-2"></div>
                        <div className="h-3 bg-muted/50 rounded w-1/2"></div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="space-y-4">
                    {contacts.slice(0, 5).map(contact => (
                      <div key={contact.id} className="flex items-center gap-3">
                        <div className="h-9 w-9 rounded-full bg-muted flex items-center justify-center overflow-hidden">
                          {contact.avatar ? (
                            <img src={contact.avatar} alt={contact.name} className="h-full w-full object-cover" />
                          ) : (
                            <span className="text-sm font-medium">{contact.name.charAt(0)}</span>
                          )}
                        </div>
                        <div>
                          <p className="text-sm font-medium">{contact.name}</p>
                          <p className="text-xs text-muted-foreground">{contact.company}</p>
                        </div>
                      </div>
                    ))}
                    {contacts.length === 0 && (
                      <div className="text-center py-4 text-muted-foreground">
                        <Users className="h-8 w-8 mx-auto mb-2 opacity-50" />
                        <p>No contacts created yet</p>
                        <Button variant="link" size="sm" className="mt-2" asChild>
                          <Link href="/contacts">Add contacts</Link>
                        </Button>
                      </div>
                    )}
                  </div>
                )}
              </CardContent>
            </Card>
            <Card>
              <CardHeader>
                <CardTitle>Sales Performance</CardTitle>
                <CardDescription>
                  Deal closure and conversion rates
                </CardDescription>
              </CardHeader>
              <CardContent>
                {isLoading ? (
                  <div className="space-y-4">
                    {[...Array(3)].map((_, i) => (
                      <div key={i} className="animate-pulse">
                        <div className="h-4 bg-muted rounded w-2/3 mb-2"></div>
                        <div className="h-3 bg-muted/50 rounded w-1/2"></div>
                        <div className="h-2 bg-muted/30 rounded-full w-full mt-2"></div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="space-y-4">
                    <div>
                      <div className="flex items-center justify-between">
                        <p className="text-sm">Win Rate</p>
                        <p className="text-sm font-medium">
                          {deals.length === 0 ? '0%' : Math.round(deals.filter(d => d.stage === 'closed').length / deals.length * 100) + '%'}
                        </p>
                      </div>
                      <div className="mt-2 h-2 w-full bg-muted rounded-full overflow-hidden">
                        <div className="h-full bg-green-500 rounded-full" style={{ width: `${deals.length === 0 ? 0 : Math.round(deals.filter(d => d.stage === 'closed').length / deals.length * 100)}%` }}></div>
                      </div>
                    </div>
                    <div>
                      <div className="flex items-center justify-between">
                        <p className="text-sm">Conversion Rate</p>
                        <p className="text-sm font-medium">
                          {deals.length === 0 ? '0%' : Math.round(deals.filter(d => d.stage !== 'qualified').length / deals.length * 100) + '%'}
                        </p>
                      </div>
                      <div className="mt-2 h-2 w-full bg-muted rounded-full overflow-hidden">
                        <div className="h-full bg-blue-500 rounded-full" style={{ width: `${deals.length === 0 ? 0 : Math.round(deals.filter(d => d.stage !== 'qualified').length / deals.length * 100)}%` }}></div>
                      </div>
                    </div>
                    <div>
                      <div className="flex items-center justify-between">
                        <p className="text-sm">Average Deal Size</p>
                        <p className="text-sm font-medium">
                          ${deals.length === 0 ? '0' : Math.round(deals.reduce((sum, d) => sum + (d.value || 0), 0) / deals.length).toLocaleString()}
                        </p>
                      </div>
                      <div className="text-xs text-muted-foreground flex items-center mt-1">
                        <ArrowUp className="h-3 w-3 mr-1 text-green-500" />
                        <span className="text-green-500">12%</span>
                        <span className="ml-1">vs last month</span>
                      </div>
                    </div>
                    <div className="mt-5 pt-3 border-t">
                      <Button variant="outline" size="sm" className="w-full" asChild>
                        <Link href="/reports">
                          <BarChart3 className="h-3.5 w-3.5 mr-2" />
                          View Full Reports
                        </Link>
                      </Button>
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </TabsContent>
      </Tabs>
    </div>
  )
} 