"use client"

import { useState } from "react"
import { DragDropContext, Droppable, Draggable } from "react-beautiful-dnd"

// Mock data for deals
const initialDeals = [
  {
    id: "deal-1",
    title: "Enterprise Software Package",
    company: "Acme Corp",
    value: "$24,000",
    contact: "John Smith",
    stage: "qualified"
  },
  {
    id: "deal-2",
    title: "Consulting Services",
    company: "TechGiant Inc",
    value: "$15,000",
    contact: "Sarah Johnson",
    stage: "qualified"
  },
  {
    id: "deal-3",
    title: "Annual Support Contract",
    company: "SmallBusiness LLC",
    value: "$8,500",
    contact: "David Wilson",
    stage: "proposal"
  },
  {
    id: "deal-4",
    title: "Premium SaaS Subscription",
    company: "MegaCorp",
    value: "$36,000",
    contact: "Michael Brown",
    stage: "negotiation"
  },
  {
    id: "deal-5",
    title: "Software Implementation",
    company: "Startup Inc",
    value: "$12,800",
    contact: "Jennifer Lee",
    stage: "proposal"
  },
  {
    id: "deal-6",
    title: "Hardware Upgrade",
    company: "Local Business",
    value: "$5,200",
    contact: "Robert Chen",
    stage: "closed"
  },
]

const stageInfo = {
  qualified: { 
    title: "Qualified", 
    color: "bg-blue-500" 
  },
  proposal: { 
    title: "Proposal", 
    color: "bg-amber-500" 
  },
  negotiation: { 
    title: "Negotiation", 
    color: "bg-violet-500" 
  },
  closed: { 
    title: "Closed Won", 
    color: "bg-green-500" 
  }
}

export default function DealsPage() {
  const [deals, setDeals] = useState(initialDeals)

  // Group deals by stage
  const dealsByStage = deals.reduce((acc, deal) => {
    if (!acc[deal.stage]) {
      acc[deal.stage] = []
    }
    acc[deal.stage].push(deal)
    return acc
  }, {} as Record<string, typeof initialDeals>)

  const onDragEnd = (result: any) => {
    const { destination, source, draggableId } = result

    // Dropped outside the list
    if (!destination) {
      return
    }

    // Dropped in the same place
    if (
      destination.droppableId === source.droppableId &&
      destination.index === source.index
    ) {
      return
    }

    // Moving within the same column
    if (source.droppableId === destination.droppableId) {
      const stageDeals = Array.from(dealsByStage[source.droppableId] || [])
      const [removed] = stageDeals.splice(source.index, 1)
      stageDeals.splice(destination.index, 0, removed)

      setDeals(
        deals.map(deal => 
          deal.id === draggableId 
            ? { ...deal } 
            : deal
        )
      )
      return
    }

    // Moving to a different column
    const sourceStageDeals = Array.from(dealsByStage[source.droppableId] || [])
    const [removed] = sourceStageDeals.splice(source.index, 1)
    const destStageDeals = Array.from(dealsByStage[destination.droppableId] || [])
    destStageDeals.splice(destination.index, 0, removed)

    setDeals(
      deals.map(deal => 
        deal.id === draggableId 
          ? { ...deal, stage: destination.droppableId } 
          : deal
      )
    )
  }

  return (
    <div className="p-6 space-y-8">
      <div className="space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Pipeline</h2>
        <p className="text-muted-foreground">
          Drag and drop deals to update their stage in the sales pipeline.
        </p>
      </div>

      <DragDropContext onDragEnd={onDragEnd}>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-6">
          {Object.entries(stageInfo).map(([stageId, stage]) => (
            <div key={stageId} className="flex flex-col h-full">
              <div className="flex items-center space-x-2 mb-3">
                <div className={`w-3 h-3 rounded-full ${stage.color}`} />
                <h3 className="font-medium">{stage.title}</h3>
                <div className="text-muted-foreground text-sm ml-auto">
                  {dealsByStage[stageId]?.length || 0} deals
                </div>
              </div>
              
              <Droppable droppableId={stageId}>
                {(provided) => (
                  <div
                    {...provided.droppableProps}
                    ref={provided.innerRef}
                    className="flex-1 bg-muted/20 rounded-lg p-2 min-h-[70vh] flex flex-col"
                  >
                    {dealsByStage[stageId]?.map((deal, index) => (
                      <Draggable
                        key={deal.id}
                        draggableId={deal.id}
                        index={index}
                      >
                        {(provided) => (
                          <div
                            ref={provided.innerRef}
                            {...provided.draggableProps}
                            {...provided.dragHandleProps}
                            className="bg-card text-card-foreground rounded-md p-3 shadow-sm border mb-2"
                          >
                            <div className="font-medium">{deal.title}</div>
                            <div className="text-sm text-muted-foreground mt-1">{deal.company}</div>
                            <div className="flex justify-between items-center mt-2">
                              <div className="text-sm font-medium">{deal.value}</div>
                              <div className="text-xs text-muted-foreground">{deal.contact}</div>
                            </div>
                          </div>
                        )}
                      </Draggable>
                    ))}
                    {provided.placeholder}
                  </div>
                )}
              </Droppable>
            </div>
          ))}
        </div>
      </DragDropContext>
    </div>
  )
} 