"use client"

import { useState } from "react"
import { CheckCircle2, ChevronDown, ChevronUp, Clock, CalendarClock } from "lucide-react"

// Mock data for tasks
const initialTasks = [
  {
    id: "task-1",
    title: "Follow up with John Smith",
    description: "Send proposal follow-up email",
    dueDate: "Today at 2:00 PM",
    priority: "high",
    related: "Acme Corp - Software Package",
    status: "pending",
    isExpanded: false,
  },
  {
    id: "task-2",
    title: "Prepare presentation for TechGiant Inc",
    description: "Create slides for the initial consultation meeting",
    dueDate: "Tomorrow at 10:00 AM",
    priority: "medium",
    related: "TechGiant Inc - Consulting Services",
    status: "pending",
    isExpanded: false,
  },
  {
    id: "task-3",
    title: "Call SmallBusiness LLC about support contract",
    description: "Discuss renewal terms and potential upgrades",
    dueDate: "May 22, 2023",
    priority: "low",
    related: "SmallBusiness LLC - Support Contract",
    status: "pending",
    isExpanded: false,
  },
  {
    id: "task-4",
    title: "Send pricing information to MegaCorp",
    description: "Detail premium tier options and volume discounts",
    dueDate: "May 20, 2023",
    priority: "high",
    related: "MegaCorp - SaaS Subscription",
    status: "completed",
    isExpanded: false,
  },
  {
    id: "task-5",
    title: "Schedule demo with Startup Inc",
    description: "Showcase new features in our platform",
    dueDate: "May 25, 2023",
    priority: "medium",
    related: "Startup Inc - Implementation",
    status: "pending",
    isExpanded: false,
  },
]

export default function TasksPage() {
  const [tasks, setTasks] = useState(initialTasks)
  
  // Toggle task completion status
  const toggleTaskStatus = (taskId: string) => {
    setTasks(tasks.map(task => 
      task.id === taskId 
        ? { ...task, status: task.status === "completed" ? "pending" : "completed" } 
        : task
    ))
  }
  
  // Toggle task details visibility
  const toggleTaskDetails = (taskId: string) => {
    setTasks(tasks.map(task => 
      task.id === taskId 
        ? { ...task, isExpanded: !task.isExpanded } 
        : task
    ))
  }

  // Count pending tasks
  const pendingTasksCount = tasks.filter(task => task.status === "pending").length

  return (
    <div className="p-6 space-y-8">
      <div className="space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Tasks</h2>
        <p className="text-muted-foreground">
          Manage your follow-ups and activities with customers and leads.
        </p>
      </div>

      <div className="flex justify-between items-center">
        <div className="flex space-x-4">
          <div className="bg-card border rounded-md p-3 flex items-center space-x-3">
            <Clock className="h-5 w-5 text-muted-foreground" />
            <div>
              <div className="text-xs text-muted-foreground">Pending Tasks</div>
              <div className="text-xl font-bold">{pendingTasksCount}</div>
            </div>
          </div>
          <div className="bg-card border rounded-md p-3 flex items-center space-x-3">
            <CalendarClock className="h-5 w-5 text-muted-foreground" />
            <div>
              <div className="text-xs text-muted-foreground">Due Today</div>
              <div className="text-xl font-bold">1</div>
            </div>
          </div>
        </div>
        <button className="bg-primary text-primary-foreground hover:bg-primary/90 px-4 py-2 rounded-md text-sm font-medium">
          Add Task
        </button>
      </div>

      <div className="space-y-4">
        {tasks.map(task => (
          <div 
            key={task.id} 
            className={`border rounded-md shadow-sm overflow-hidden ${
              task.status === "completed" ? "bg-muted/30" : "bg-card"
            }`}
          >
            <div 
              className="p-4 flex items-center cursor-pointer"
              onClick={() => toggleTaskDetails(task.id)}
            >
              <button 
                onClick={(e) => {
                  e.stopPropagation();
                  toggleTaskStatus(task.id);
                }}
                className={`mr-3 flex-shrink-0 ${
                  task.status === "completed" ? "text-green-500" : "text-muted-foreground"
                }`}
              >
                <CheckCircle2 className="h-5 w-5" />
              </button>
              <div className="flex-1 min-w-0">
                <div className={`font-medium ${task.status === "completed" ? "line-through text-muted-foreground" : ""}`}>
                  {task.title}
                </div>
                <div className="text-xs text-muted-foreground mt-1">
                  Due: {task.dueDate}
                </div>
              </div>
              <div className={`px-2 py-1 rounded-full text-xs font-medium mr-3 ${
                task.priority === "high" 
                  ? "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300"
                  : task.priority === "medium"
                    ? "bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-300"
                    : "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300"
              }`}>
                {task.priority.charAt(0).toUpperCase() + task.priority.slice(1)}
              </div>
              {task.isExpanded ? <ChevronUp className="h-5 w-5" /> : <ChevronDown className="h-5 w-5" />}
            </div>
            
            {task.isExpanded && (
              <div className="px-4 pb-4 pt-2 border-t">
                <div className="text-sm mb-2">{task.description}</div>
                <div className="text-xs text-muted-foreground">
                  Related: {task.related}
                </div>
              </div>
            )}
          </div>
        ))}
      </div>
    </div>
  )
} 