"use client"

import { useState } from "react"
import { 
  Check, 
  Clock, 
  Mail, 
  MoreHorizontal, 
  Plus, 
  Search, 
  Shield, 
  UserPlus, 
  X
} from "lucide-react"
import Image from "next/image"

import { users } from "@/lib/mock-data"
import { User } from "@/types/crm"
import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"

export default function TeamPage() {
  const [searchTerm, setSearchTerm] = useState("")
  const [showInviteForm, setShowInviteForm] = useState(false)
  const [inviteEmail, setInviteEmail] = useState("")
  const [selectedRole, setSelectedRole] = useState<User['role']>("user")
  
  // Filter users based on search term
  const filteredUsers = users.filter(user => 
    user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.role.toLowerCase().includes(searchTerm.toLowerCase())
  )

  const handleInvite = () => {
    // In a real app, this would send an invitation to the email
    console.log(`Invite sent to ${inviteEmail} with role ${selectedRole}`)
    setInviteEmail("")
    setSelectedRole("user")
    setShowInviteForm(false)
  }

  const getRoleBadgeClass = (role: User['role']) => {
    switch (role) {
      case 'admin':
        return "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300"
      case 'manager':
        return "bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-300"
      default:
        return "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300"
    }
  }

  const getRoleIcon = (role: User['role']) => {
    switch (role) {
      case 'admin':
        return <Shield className="h-3 w-3 mr-1" />
      case 'manager':
        return <Check className="h-3 w-3 mr-1" />
      default:
        return <Clock className="h-3 w-3 mr-1" />
    }
  }

  return (
    <div className="p-6 space-y-8">
      <div className="space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Team</h2>
        <p className="text-muted-foreground">
          Manage your team members and their roles.
        </p>
      </div>

      <div className="flex justify-between items-center">
        <div className="relative w-full max-w-sm">
          <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
          <Input
            type="text"
            placeholder="Search team members..."
            className="pl-9"
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
          />
        </div>
        <Button onClick={() => setShowInviteForm(true)}>
          <UserPlus className="h-4 w-4 mr-2" />
          Invite User
        </Button>
      </div>

      {showInviteForm && (
        <Card className="border-dashed">
          <CardContent className="pt-6">
            <div className="flex justify-between items-center mb-4">
              <h3 className="font-semibold">Invite New Team Member</h3>
              <Button 
                variant="ghost" 
                size="sm"
                onClick={() => setShowInviteForm(false)}
              >
                <X className="h-4 w-4" />
              </Button>
            </div>
            <div className="space-y-4">
              <div>
                <label className="text-sm font-medium mb-1 block">Email Address</label>
                <Input
                  type="email"
                  placeholder="colleague@example.com"
                  value={inviteEmail}
                  onChange={(e) => setInviteEmail(e.target.value)}
                />
              </div>
              <div>
                <label className="text-sm font-medium mb-1 block">Role</label>
                <div className="flex space-x-2">
                  {(['user', 'manager', 'admin'] as const).map(role => (
                    <Button
                      key={role}
                      variant={selectedRole === role ? "default" : "outline"}
                      size="sm"
                      onClick={() => setSelectedRole(role)}
                      className="capitalize"
                    >
                      {role}
                    </Button>
                  ))}
                </div>
              </div>
              <div className="flex justify-end">
                <Button 
                  onClick={handleInvite}
                  disabled={!inviteEmail.includes('@')}
                >
                  <Mail className="h-4 w-4 mr-2" />
                  Send Invitation
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      <div className="rounded-md border">
        <div className="p-4 flex items-center text-sm font-medium bg-muted/50">
          <div className="w-10"></div>
          <div className="flex-1">Name</div>
          <div className="w-[180px]">Role</div>
          <div className="w-[150px]">Status</div>
          <div className="w-[40px]"></div>
        </div>
        <div className="divide-y">
          {filteredUsers.map(user => (
            <div key={user.id} className="p-4 flex items-center">
              <div className="w-10">
                {user.avatar ? (
                  <div className="h-8 w-8 rounded-full overflow-hidden relative">
                    <Image
                      src={user.avatar}
                      alt={user.name}
                      fill
                      className="object-cover"
                    />
                  </div>
                ) : (
                  <div className="h-8 w-8 rounded-full bg-muted flex items-center justify-center">
                    <span className="text-xs font-medium">{user.name.charAt(0)}</span>
                  </div>
                )}
              </div>
              <div className="flex-1">
                <p className="font-medium">{user.name}</p>
                <p className="text-sm text-muted-foreground">{user.email}</p>
              </div>
              <div className="w-[180px]">
                <span className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium ${getRoleBadgeClass(user.role)}`}>
                  {getRoleIcon(user.role)}
                  <span className="capitalize">{user.role}</span>
                </span>
              </div>
              <div className="w-[150px]">
                <div className="flex items-center">
                  <div className={`h-2 w-2 rounded-full mr-2 ${user.lastActive === 'Online' ? 'bg-green-500' : 'bg-amber-500'}`}></div>
                  <span className="text-sm">{user.lastActive}</span>
                </div>
              </div>
              <div className="w-[40px] text-right">
                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                      <MoreHorizontal className="h-4 w-4" />
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent align="end">
                    <DropdownMenuLabel>Actions</DropdownMenuLabel>
                    <DropdownMenuSeparator />
                    <DropdownMenuItem>View Profile</DropdownMenuItem>
                    <DropdownMenuItem>Edit Permissions</DropdownMenuItem>
                    <DropdownMenuItem>Reset Password</DropdownMenuItem>
                    <DropdownMenuSeparator />
                    <DropdownMenuItem className="text-destructive">Remove User</DropdownMenuItem>
                  </DropdownMenuContent>
                </DropdownMenu>
              </div>
            </div>
          ))}
        </div>
      </div>

      <Card>
        <CardContent className="pt-6">
          <h3 className="font-semibold mb-4">About User Roles</h3>
          <div className="space-y-4">
            <div className="flex items-start gap-3">
              <div className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium whitespace-nowrap ${getRoleBadgeClass('admin')}`}>
                {getRoleIcon('admin')}
                <span>Admin</span>
              </div>
              <p className="text-sm text-muted-foreground">Full access to all settings and features. Can manage users and billing.</p>
            </div>
            <div className="flex items-start gap-3">
              <div className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium whitespace-nowrap ${getRoleBadgeClass('manager')}`}>
                {getRoleIcon('manager')}
                <span>Manager</span>
              </div>
              <p className="text-sm text-muted-foreground">Can manage team tasks, deals, and contacts. Limited access to settings.</p>
            </div>
            <div className="flex items-start gap-3">
              <div className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium whitespace-nowrap ${getRoleBadgeClass('user')}`}>
                {getRoleIcon('user')}
                <span>User</span>
              </div>
              <p className="text-sm text-muted-foreground">Basic access to assigned deals and contacts. Cannot modify system settings.</p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  )
} 