"use client";

import { useState } from 'react';
import { getSupabaseClient } from '@/lib/supabase';
import Link from 'next/link';
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";

export default function ResetPasswordPage() {
  const [email, setEmail] = useState('');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState(false);

  async function handleReset(e: React.FormEvent<HTMLFormElement>) {
    e.preventDefault();
    setError(null);
    setSuccess(false);
    setLoading(true);
    try {
      const supabase = getSupabaseClient();
      const { error } = await supabase.auth.resetPasswordForEmail(email, {
        redirectTo: `${window.location.origin}/auth/update-password`,
      });
      if (error) {
        setError(error.message);
        return;
      }
      setSuccess(true);
    } catch (err) {
      setError('Unexpected error occurred. Please try again.');
    } finally {
      setLoading(false);
    }
  }

  return (
    <>
      <h1 className="text-2xl font-bold">Reset your password</h1>
      <p className="text-sm text-muted-foreground">
        Enter your email to receive a password reset link
      </p>

      {success ? (
        <div className="mt-6 space-y-4">
          <div className="bg-green-50 text-green-800 p-4 rounded-md text-sm">
            We've sent you an email with a password reset link. Please check your inbox.
          </div>
          <Button asChild className="w-full">
            <Link href="/auth/login">
              Return to login
            </Link>
          </Button>
        </div>
      ) : (
        <form className="space-y-4 mt-6" onSubmit={handleReset}>
          <div className="space-y-2">
            <Label htmlFor="email">Email</Label>
            <Input
              id="email"
              type="email"
              required
              placeholder="name@example.com"
              value={email}
              onChange={(e) => setEmail(e.target.value)}
            />
          </div>
          {error && (
            <div className="text-sm text-destructive text-center bg-destructive/10 p-2 rounded-md">
              {error}
            </div>
          )}
          <Button 
            type="submit" 
            className="w-full"
            disabled={loading}
          >
            {loading ? "Sending..." : "Send reset link"}
          </Button>
        </form>
      )}
      
      <div className="mt-4 text-center text-sm">
        Remember your password?{" "}
        <Link href="/auth/login" className="text-primary font-medium hover:underline">
          Sign in
        </Link>
      </div>
    </>
  );
} 