"use client"

import { format } from "date-fns"
import { 
  CalendarClock, 
  CheckCircle2, 
  CircleUser, 
  Clock, 
  Contact, 
  DollarSign,
  FilePlus,
  FileText,
  MessageSquare,
  Pencil, 
  PieChart, 
  Plus
} from "lucide-react"

import { ActivityLog } from "@/types/crm"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"

// Helper function to get icon based on activity type
const getActivityIcon = (type: ActivityLog['type']) => {
  switch (type) {
    case 'contact_created':
      return <Plus className="h-4 w-4 text-green-500" />
    case 'contact_updated':
      return <Pencil className="h-4 w-4 text-blue-500" />
    case 'deal_created':
      return <FilePlus className="h-4 w-4 text-violet-500" />
    case 'deal_updated':
      return <FileText className="h-4 w-4 text-amber-500" />
    case 'deal_stage_changed':
      return <PieChart className="h-4 w-4 text-pink-500" />
    case 'task_created':
      return <Clock className="h-4 w-4 text-indigo-500" />
    case 'task_completed':
      return <CheckCircle2 className="h-4 w-4 text-emerald-500" />
    default:
      return <MessageSquare className="h-4 w-4 text-gray-500" />
  }
}

// Helper function to get icon based on entity type
const getEntityIcon = (entityType: ActivityLog['entityType']) => {
  switch (entityType) {
    case 'contact':
      return <Contact className="h-4 w-4" />
    case 'deal':
      return <DollarSign className="h-4 w-4" />
    case 'task':
      return <CalendarClock className="h-4 w-4" />
    case 'company':
      return <CircleUser className="h-4 w-4" />
    default:
      return <MessageSquare className="h-4 w-4" />
  }
}

interface ActivityLogComponentProps {
  activities: ActivityLog[]
  limit?: number
}

export function ActivityLogComponent({ activities, limit = 5 }: ActivityLogComponentProps) {
  // Sort activities by timestamp (newest first) and limit the number
  const sortedActivities = [...activities]
    .sort((a, b) => new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime())
    .slice(0, limit)

  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-lg font-medium">Recent Activities</CardTitle>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {sortedActivities.length === 0 ? (
            <p className="text-sm text-muted-foreground text-center py-4">No recent activities</p>
          ) : (
            <div className="space-y-4">
              {sortedActivities.map((activity) => (
                <div key={activity.id} className="flex items-start space-x-3 text-sm">
                  <div className="h-8 w-8 rounded-full bg-muted flex items-center justify-center flex-shrink-0">
                    {getActivityIcon(activity.type)}
                  </div>
                  <div className="flex-1 space-y-1">
                    <p className="font-medium">{activity.description}</p>
                    <div className="flex items-center space-x-2 text-xs text-muted-foreground">
                      <span>{format(new Date(activity.timestamp), 'MMM d, yyyy h:mm a')}</span>
                      <span>•</span>
                      <div className="flex items-center">
                        {getEntityIcon(activity.entityType)}
                        <span className="ml-1 capitalize">{activity.entityType}</span>
                      </div>
                      <span>•</span>
                      <span>{activity.user}</span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  )
} 