"use client"

import { 
  Activity,
  CalendarClock, 
  CheckCircle2, 
  MessageSquare, 
  CreditCard, 
  User, 
  UserPlus, 
  Building, 
  Mail,
  Phone
} from "lucide-react"
import { format, formatDistanceToNow } from "date-fns"
import { ActivityLog } from "@/types/crm"

type ActivityItemProps = {
  activity: ActivityLog
}

export function ActivityItem({ activity }: ActivityItemProps) {
  const getActivityIcon = (type: string) => {
    switch (type) {
      case 'task':
        return <CheckCircle2 className="h-5 w-5 text-green-500" />
      case 'call':
        return <Phone className="h-5 w-5 text-blue-500" />
      case 'email':
        return <Mail className="h-5 w-5 text-amber-500" />
      case 'meeting':
        return <CalendarClock className="h-5 w-5 text-purple-500" />
      case 'note':
        return <MessageSquare className="h-5 w-5 text-gray-500" />
      case 'deal_update':
        return <CreditCard className="h-5 w-5 text-emerald-500" />
      case 'contact_created':
        return <UserPlus className="h-5 w-5 text-blue-500" />
      default:
        return <Activity className="h-5 w-5 text-gray-500" />
    }
  }
  
  const formatTimestamp = (timestamp: string) => {
    try {
      const date = new Date(timestamp)
      const isToday = new Date().toDateString() === date.toDateString()
      
      if (isToday) {
        return formatDistanceToNow(date, { addSuffix: true })
      }
      
      return format(date, "MMM d, yyyy 'at' h:mm a")
    } catch (error) {
      return timestamp
    }
  }
  
  return (
    <div className="flex items-start space-x-4">
      <div className="h-9 w-9 rounded-full bg-muted flex items-center justify-center overflow-hidden">
        {activity.user && activity.user.avatar ? (
          <img 
            src={activity.user.avatar} 
            alt={activity.user.name || 'User'} 
            className="h-full w-full object-cover" 
          />
        ) : (
          <User className="h-5 w-5 text-muted-foreground" />
        )}
      </div>
      <div className="space-y-1 flex-1">
        <div className="flex items-center gap-2">
          {getActivityIcon(activity.type)}
          <p className="text-sm font-medium">{activity.description}</p>
        </div>
        <div className="flex items-center text-xs text-muted-foreground">
          <span>{activity.user ? activity.user.name : 'Unknown User'}</span>
          <span className="mx-1">•</span>
          <span>{formatTimestamp(activity.timestamp)}</span>
        </div>
      </div>
    </div>
  )
} 