"use client";

import { useState, useTransition } from 'react';
import { getSupabaseClient } from '@/lib/supabase';
import { useRouter } from 'next/navigation';
import { PencilIcon, XIcon } from 'lucide-react';
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardHeader, CardTitle, CardFooter } from "@/components/ui/card";
import type { User } from '@supabase/supabase-js';

interface ProfileCardProps {
  user: User;
  profile: {
    id?: string;
    name?: string | null;
    email?: string | null;
    avatar_url?: string | null;
    role?: string | null;
  } | null;
}

export default function ProfileCard({ user, profile }: ProfileCardProps) {
  const router = useRouter();
  const [isPending, startTransition] = useTransition();
  const [isEditing, setIsEditing] = useState(false);
  const [username, setUsername] = useState(profile?.name || '');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  async function handleUpdateProfile(e: React.FormEvent<HTMLFormElement>) {
    e.preventDefault();
    setError(null);
    setLoading(true);
    
    try {
      console.log('Submitting profile update:', { userId: user.id, username });
      
      const supabase = getSupabaseClient();
      
      const { data: updateResult, error: updateError } = await supabase
        .from('profiles')
        .upsert({ 
          id: user.id,
          name: username,
          email: user.email || '',
          updated_at: new Date().toISOString()
        }, {
          onConflict: 'id'
        });

      console.log('Upsert result:', updateResult, updateError);

      if (updateError) {
        setError(updateError.message);
        return;
      }
      
      setIsEditing(false);
      startTransition(() => {
        router.refresh();
      });
    } catch (err) {
      setError('Unexpected error.');
      console.error('Profile update error:', err);
    } finally {
      setLoading(false);
    }
  }

  async function handleLogout() {
    setLoading(true);
    try {
      const supabase = getSupabaseClient();
      await supabase.auth.signOut();
      router.push('/auth/login');
    } catch (err) {
      setError('Error signing out');
      console.error('Logout error:', err);
    } finally {
      setLoading(false);
    }
  }

  return (
    <div className="w-full max-w-md bg-white rounded-lg shadow-md p-6 border border-gray-200">
      <div className="flex items-center justify-between mb-4">
        <h2 className="text-xl font-bold">Your Profile</h2>
        {!isEditing ? (
          <button
            onClick={() => setIsEditing(true)}
            className="flex items-center gap-2 px-3 py-1.5 text-sm rounded-lg bg-gray-100 hover:bg-gray-200 transition-colors"
          >
            <PencilIcon className="w-4 h-4" />
            Edit Profile
          </button>
        ) : (
          <button
            onClick={() => {
              setIsEditing(false);
              setUsername(profile?.name || '');
              setError(null);
            }}
            className="flex items-center gap-2 px-3 py-1.5 text-sm rounded-lg bg-gray-100 hover:bg-gray-200 transition-colors"
          >
            <XIcon className="w-4 h-4" />
            Cancel
          </button>
        )}
      </div>
      
      <div className="mb-6 text-sm text-center">
        Logged in as <span className="font-semibold text-blue-600">{user.email}</span>
      </div>

      {!isEditing ? (
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-500 mb-1">Username</label>
            <div className="px-4 py-2 bg-gray-50 rounded-lg border border-gray-200">
              {profile?.name || 'Not set'}
            </div>
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-500 mb-1">Email</label>
            <div className="px-4 py-2 bg-gray-50 rounded-lg border border-gray-200">
              {user.email || 'Not set'}
            </div>
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-500 mb-1">Role</label>
            <div className="px-4 py-2 bg-gray-50 rounded-lg border border-gray-200">
              {profile?.role || 'user'}
            </div>
          </div>
        </div>
      ) : (
        <form className="space-y-4" onSubmit={handleUpdateProfile}>
          <div className="space-y-2">
            <label className="block text-sm font-medium">Username</label>
            <input
              type="text"
              value={username}
              onChange={(e) => setUsername(e.target.value)}
              minLength={3}
              maxLength={20}
              required
              className="w-full px-4 py-2 bg-gray-50 border border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
              placeholder="username"
            />
          </div>
          <div className="space-y-2">
            <label className="block text-sm font-medium">Email</label>
            <input
              type="email"
              value={user.email || ''}
              disabled
              className="w-full px-4 py-2 bg-gray-100 border border-gray-200 rounded-lg cursor-not-allowed"
              placeholder="Email"
            />
            <p className="text-xs text-gray-500">Email cannot be changed</p>
          </div>
          
          {error && <div className="text-red-600 text-sm text-center">{error}</div>}
          
          <button
            type="submit"
            className="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold rounded-lg px-4 py-2 mt-2 transition-colors disabled:opacity-70"
            disabled={loading || isPending}
          >
            {loading ? 'Saving...' : 'Save changes'}
          </button>
        </form>
      )}

      <button
        className="w-full mt-6 bg-gray-50 hover:bg-gray-100 text-gray-900 font-medium rounded-lg px-4 py-2 transition-colors disabled:opacity-70"
        onClick={handleLogout}
        disabled={loading || isPending}
      >
        Sign out
      </button>
    </div>
  );
} 