"use client";

import { useState, useEffect } from "react";
import { 
  Bell, 
  Building, 
  CreditCard, 
  Save, 
  Shield, 
  User as UserIcon
} from "lucide-react";
import { 
  Card, 
  CardContent, 
  CardDescription, 
  CardFooter, 
  CardHeader, 
  CardTitle 
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import ProfileCard from "@/components/profile/ProfileCard";
import { User } from "@supabase/supabase-js";

interface ProfileData {
  username?: string;
  full_name?: string | null;
  avatar_url?: string | null;
  website?: string | null;
  created_at?: string | null;
  updated_at?: string | null;
}

interface SettingsContentProps {
  user: User;
  profile: ProfileData | null;
}

export default function SettingsContent({ user, profile }: SettingsContentProps) {
  const [activeTab, setActiveTab] = useState<"profile" | "notifications" | "company" | "security" | "billing">("profile");

  // Add debugging information
  useEffect(() => {
    console.log("Client side SettingsContent props:", { 
      userExists: !!user,
      profileExists: !!profile,
      userId: user?.id
    });
  }, [user, profile]);

  // Function to render the active tab content
  const renderTabContent = () => {
    switch (activeTab) {
      case "profile":
        return <ProfileCard user={user} profile={profile} />;
      
      case "notifications":
        return (
          <Card>
            <CardHeader>
              <CardTitle>Notification Preferences</CardTitle>
              <CardDescription>
                Configure how and when you receive notifications
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex justify-between items-center">
                  <div>
                    <p className="font-medium">Email Notifications</p>
                    <p className="text-sm text-muted-foreground">
                      Receive email updates about your account activity
                    </p>
                  </div>
                  <div className="relative inline-flex h-6 w-11 items-center rounded-full bg-primary">
                    <span className="absolute mx-0.5 h-5 w-5 rounded-full bg-white transition-transform translate-x-5"></span>
                  </div>
                </div>
                
                <div className="flex justify-between items-center">
                  <div>
                    <p className="font-medium">Task Reminders</p>
                    <p className="text-sm text-muted-foreground">
                      Get reminders about upcoming and overdue tasks
                    </p>
                  </div>
                  <div className="relative inline-flex h-6 w-11 items-center rounded-full bg-primary">
                    <span className="absolute mx-0.5 h-5 w-5 rounded-full bg-white transition-transform translate-x-5"></span>
                  </div>
                </div>
                
                <div className="flex justify-between items-center">
                  <div>
                    <p className="font-medium">Deal Updates</p>
                    <p className="text-sm text-muted-foreground">
                      Notifications when deal status changes
                    </p>
                  </div>
                  <div className="relative inline-flex h-6 w-11 items-center rounded-full bg-primary">
                    <span className="absolute mx-0.5 h-5 w-5 rounded-full bg-white transition-transform translate-x-5"></span>
                  </div>
                </div>
                
                <div className="flex justify-between items-center">
                  <div>
                    <p className="font-medium">Marketing Communications</p>
                    <p className="text-sm text-muted-foreground">
                      Receive updates about new features and promotions
                    </p>
                  </div>
                  <div className="relative inline-flex h-6 w-11 items-center rounded-full bg-muted">
                    <span className="absolute mx-0.5 h-5 w-5 rounded-full bg-white transition-transform"></span>
                  </div>
                </div>
              </div>
            </CardContent>
            <CardFooter className="justify-end">
              <Button>
                <Save className="h-4 w-4 mr-2" />
                Save Preferences
              </Button>
            </CardFooter>
          </Card>
        );
        
      case "company":
        return (
          <Card>
            <CardHeader>
              <CardTitle>Company Settings</CardTitle>
              <CardDescription>
                Update your company information and preferences
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-2">
                <Label htmlFor="company-name">Company Name</Label>
                <Input id="company-name" defaultValue="Acme Inc." />
              </div>
              <div className="space-y-2">
                <Label htmlFor="company-website">Website</Label>
                <Input id="company-website" defaultValue="https://acme.com" />
              </div>
              <div className="space-y-2">
                <Label htmlFor="company-address">Address</Label>
                <Input id="company-address" defaultValue="123 Main St, San Francisco, CA 94103" />
              </div>
              <div className="space-y-2">
                <Label htmlFor="company-industry">Industry</Label>
                <Input id="company-industry" defaultValue="Technology" />
              </div>
            </CardContent>
            <CardFooter className="justify-end">
              <Button>
                <Save className="h-4 w-4 mr-2" />
                Save Company Settings
              </Button>
            </CardFooter>
          </Card>
        );
        
      case "security":
        return (
          <Card>
            <CardHeader>
              <CardTitle>Security Settings</CardTitle>
              <CardDescription>
                Manage your password and security preferences
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-2">
                <Label htmlFor="current-password">Current Password</Label>
                <Input id="current-password" type="password" />
              </div>
              <div className="space-y-2">
                <Label htmlFor="new-password">New Password</Label>
                <Input id="new-password" type="password" />
              </div>
              <div className="space-y-2">
                <Label htmlFor="confirm-password">Confirm New Password</Label>
                <Input id="confirm-password" type="password" />
              </div>
              
              <div className="pt-4">
                <h3 className="font-medium mb-2">Two-Factor Authentication</h3>
                <div className="flex items-center justify-between">
                  <p className="text-sm text-muted-foreground">
                    Add an extra layer of security to your account
                  </p>
                  <Button variant="outline">Enable 2FA</Button>
                </div>
              </div>
            </CardContent>
            <CardFooter className="justify-end">
              <Button>
                <Save className="h-4 w-4 mr-2" />
                Update Password
              </Button>
            </CardFooter>
          </Card>
        );
        
      case "billing":
        return (
          <Card>
            <CardHeader>
              <CardTitle>Billing Information</CardTitle>
              <CardDescription>
                Manage your billing details and subscription
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="p-4 rounded-md bg-muted">
                <div className="flex justify-between items-center">
                  <div>
                    <p className="font-medium">Current Plan: Professional</p>
                    <p className="text-sm text-muted-foreground">$29/month, billed annually</p>
                  </div>
                  <Button variant="outline">Change Plan</Button>
                </div>
              </div>
              
              <div className="space-y-2">
                <Label htmlFor="card-name">Name on Card</Label>
                <Input id="card-name" defaultValue="John Doe" />
              </div>
              <div className="space-y-2">
                <Label htmlFor="card-number">Card Number</Label>
                <Input id="card-number" defaultValue="•••• •••• •••• 4242" />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="expiry">Expiry Date</Label>
                  <Input id="expiry" defaultValue="04/24" />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="cvc">CVC</Label>
                  <Input id="cvc" defaultValue="•••" />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="billing-address">Billing Address</Label>
                <Input id="billing-address" defaultValue="123 Main St, San Francisco, CA 94103" />
              </div>
            </CardContent>
            <CardFooter className="justify-end">
              <Button>
                <Save className="h-4 w-4 mr-2" />
                Update Billing Information
              </Button>
            </CardFooter>
          </Card>
        );
      
      default:
        return null;
    }
  };

  return (
    <div className="container py-10">
      <div className="mb-10">
        <h1 className="text-3xl font-bold tracking-tight">Settings</h1>
        <p className="text-muted-foreground">
          Manage your account settings and preferences
        </p>
      </div>
      
      <div className="grid md:grid-cols-[200px_1fr] lg:grid-cols-[250px_1fr] gap-8">
        <nav className="space-y-1">
          <Button
            variant={activeTab === "profile" ? "default" : "ghost"}
            className="w-full justify-start mb-1"
            onClick={() => setActiveTab("profile")}
          >
            <UserIcon className="h-4 w-4 mr-3" />
            Profile
          </Button>
          <Button
            variant={activeTab === "notifications" ? "default" : "ghost"}
            className="w-full justify-start mb-1"
            onClick={() => setActiveTab("notifications")}
          >
            <Bell className="h-4 w-4 mr-3" />
            Notifications
          </Button>
          <Button
            variant={activeTab === "company" ? "default" : "ghost"}
            className="w-full justify-start mb-1"
            onClick={() => setActiveTab("company")}
          >
            <Building className="h-4 w-4 mr-3" />
            Company
          </Button>
          <Button
            variant={activeTab === "security" ? "default" : "ghost"}
            className="w-full justify-start mb-1"
            onClick={() => setActiveTab("security")}
          >
            <Shield className="h-4 w-4 mr-3" />
            Security
          </Button>
          <Button
            variant={activeTab === "billing" ? "default" : "ghost"}
            className="w-full justify-start"
            onClick={() => setActiveTab("billing")}
          >
            <CreditCard className="h-4 w-4 mr-3" />
            Billing
          </Button>
        </nav>
        
        <div>
          {renderTabContent()}
        </div>
      </div>
    </div>
  );
} 