import { supabase, getCurrentUser } from './supabase';
import { Company, Contact, Deal, Task, ActivityLog } from '@/types/crm';

// Companies
export async function getCompanies(): Promise<Company[]> {
  const { data, error } = await supabase
    .from('companies')
    .select('*')
    .order('name');
    
  if (error) {
    console.error('Error fetching companies:', error);
    return [];
  }
  
  return data.map(company => ({
    id: company.id,
    name: company.name,
    industry: company.industry || '',
    website: company.website || '',
    size: company.size || '',
    location: company.location || '',
    logo: company.logo_url || '',
    createdAt: company.created_at,
  }));
}

export async function getCompany(id: string): Promise<Company | null> {
  const { data, error } = await supabase
    .from('companies')
    .select('*')
    .eq('id', id)
    .single();
    
  if (error || !data) {
    console.error('Error fetching company:', error);
    return null;
  }
  
  return {
    id: data.id,
    name: data.name,
    industry: data.industry || '',
    website: data.website || '',
    size: data.size || '',
    location: data.location || '',
    logo: data.logo_url || '',
    createdAt: data.created_at,
  };
}

export async function createCompany(company: Partial<Company>): Promise<Company | null> {
  const user = await getCurrentUser();
  if (!user) return null;
  
  const { data, error } = await supabase
    .from('companies')
    .insert({
      name: company.name || '',
      industry: company.industry,
      website: company.website,
      size: company.size,
      location: company.location,
      logo_url: company.logo,
      created_by: user.id,
    })
    .select()
    .single();
    
  if (error || !data) {
    console.error('Error creating company:', error);
    return null;
  }
  
  return {
    id: data.id,
    name: data.name,
    industry: data.industry || '',
    website: data.website || '',
    size: data.size || '',
    location: data.location || '',
    logo: data.logo_url || '',
    createdAt: data.created_at,
  };
}

// Contacts
export async function getContacts(): Promise<Contact[]> {
  const { data, error } = await supabase
    .from('contacts')
    .select(`
      *,
      companies(name)
    `)
    .order('name');
    
  if (error) {
    console.error('Error fetching contacts:', error);
    return [];
  }
  
  return data.map(contact => ({
    id: contact.id,
    name: contact.name,
    email: contact.email || '',
    phone: contact.phone || '',
    company: contact.companies?.name || '',
    companyId: contact.company_id || '',
    position: contact.position || '',
    status: contact.status,
    notes: contact.notes || '',
    avatar: contact.avatar_url || '',
    tags: contact.tags || [],
    createdAt: contact.created_at,
  }));
}

export async function getContact(id: string): Promise<Contact | null> {
  const { data, error } = await supabase
    .from('contacts')
    .select(`
      *,
      companies(name)
    `)
    .eq('id', id)
    .single();
    
  if (error || !data) {
    console.error('Error fetching contact:', error);
    return null;
  }
  
  return {
    id: data.id,
    name: data.name,
    email: data.email || '',
    phone: data.phone || '',
    company: data.companies?.name || '',
    companyId: data.company_id || '',
    position: data.position || '',
    status: data.status,
    notes: data.notes || '',
    avatar: data.avatar_url || '',
    tags: data.tags || [],
    createdAt: data.created_at,
  };
}

export async function createContact(contact: Partial<Contact>): Promise<Contact | null> {
  const user = await getCurrentUser();
  if (!user) return null;
  
  const { data, error } = await supabase
    .from('contacts')
    .insert({
      name: contact.name || '',
      email: contact.email,
      phone: contact.phone,
      company_id: contact.companyId,
      position: contact.position,
      status: contact.status || 'active',
      notes: contact.notes,
      avatar_url: contact.avatar,
      tags: contact.tags,
      created_by: user.id,
    })
    .select()
    .single();
    
  if (error || !data) {
    console.error('Error creating contact:', error);
    return null;
  }
  
  // Create an activity record for this creation
  await supabase.from('activities').insert({
    type: 'contact_created',
    description: `Contact created: ${contact.name}`,
    user_id: user.id,
    related_id: data.id,
    related_type: 'contact'
  });
  
  return {
    id: data.id,
    name: data.name,
    email: data.email || '',
    phone: data.phone || '',
    company: '',
    companyId: data.company_id || '',
    position: data.position || '',
    status: data.status,
    notes: data.notes || '',
    avatar: data.avatar_url || '',
    tags: data.tags || [],
    createdAt: data.created_at,
  };
}

// Deals
export async function getDeals(): Promise<Deal[]> {
  const { data, error } = await supabase
    .from('deals')
    .select(`
      *,
      companies(name),
      contacts(name),
      pipeline_stages(id, title, color)
    `)
    .order('created_at', { ascending: false });
    
  if (error) {
    console.error('Error fetching deals:', error);
    return [];
  }
  
  return data.map(deal => ({
    id: deal.id,
    title: deal.title,
    value: deal.value || 0,
    company: deal.companies?.name || '',
    companyId: deal.company_id || '',
    contact: deal.contacts?.name || '',
    contactId: deal.contact_id || '',
    stage: deal.pipeline_stages?.id || '',
    stageTitle: deal.pipeline_stages?.title || '',
    stageColor: deal.pipeline_stages?.color || '',
    priority: deal.priority,
    description: deal.description || '',
    createdAt: deal.created_at,
    updatedAt: deal.updated_at,
    closedAt: deal.closed_at,
  }));
}

export async function createDeal(deal: Partial<Deal>): Promise<Deal | null> {
  const user = await getCurrentUser();
  if (!user) return null;
  
  const { data, error } = await supabase
    .from('deals')
    .insert({
      title: deal.title || '',
      value: deal.value,
      company_id: deal.companyId,
      contact_id: deal.contactId,
      stage_id: deal.stage || 'qualified',
      priority: deal.priority || 'medium',
      description: deal.description,
      owner_id: user.id,
    })
    .select()
    .single();
    
  if (error || !data) {
    console.error('Error creating deal:', error);
    return null;
  }
  
  // Create an activity for this creation
  await supabase.from('activities').insert({
    type: 'deal_update',
    description: `Deal created: ${deal.title}`,
    user_id: user.id,
    related_id: data.id,
    related_type: 'deal'
  });
  
  // Fetch the stage info
  const { data: stageData } = await supabase
    .from('pipeline_stages')
    .select('*')
    .eq('id', data.stage_id)
    .single();
  
  return {
    id: data.id,
    title: data.title,
    value: data.value || 0,
    company: '',
    companyId: data.company_id || '',
    contact: '',
    contactId: data.contact_id || '',
    stage: data.stage_id,
    stageTitle: stageData?.title || '',
    stageColor: stageData?.color || '',
    priority: data.priority,
    description: data.description || '',
    createdAt: data.created_at,
    updatedAt: data.updated_at,
    closedAt: data.closed_at,
  };
}

export async function updateDealStage(dealId: string, stageId: string): Promise<boolean> {
  const user = await getCurrentUser();
  if (!user) return false;
  
  const { error } = await supabase
    .from('deals')
    .update({ stage_id: stageId, updated_at: new Date().toISOString() })
    .eq('id', dealId);
    
  if (error) {
    console.error('Error updating deal stage:', error);
    return false;
  }
  
  // Get stage name
  const { data: stageData } = await supabase
    .from('pipeline_stages')
    .select('title')
    .eq('id', stageId)
    .single();
    
  // Get deal title
  const { data: dealData } = await supabase
    .from('deals')
    .select('title')
    .eq('id', dealId)
    .single();
  
  // Create an activity for this update
  await supabase.from('activities').insert({
    type: 'deal_update',
    description: `Deal '${dealData?.title}' moved to '${stageData?.title}' stage`,
    user_id: user.id,
    related_id: dealId,
    related_type: 'deal',
    metadata: { newStageId: stageId }
  });
  
  return true;
}

// Pipeline Stages
export async function getPipelineStages() {
  const { data, error } = await supabase
    .from('pipeline_stages')
    .select('*')
    .order('order');
    
  if (error) {
    console.error('Error fetching pipeline stages:', error);
    return [];
  }
  
  return data;
}

// Activities
export async function getActivities(limit = 10): Promise<ActivityLog[]> {
  const { data, error } = await supabase
    .from('activities_with_users')
    .select('*')
    .order('timestamp', { ascending: false })
    .limit(limit);
    
  if (error) {
    console.error('Error fetching activities:', error);
    return [];
  }
  
  return data.map(activity => {
    return {
      id: activity.id,
      type: activity.type,
      description: activity.description,
      timestamp: activity.timestamp,
      user: {
        id: activity.user_id,
        name: activity.user_name || activity.user_email?.split('@')[0] || 'Unknown',
        email: activity.user_email || '',
        avatar: activity.user_avatar || '',
      },
      relatedId: activity.related_id || '',
      relatedType: activity.related_type || '',
    };
  });
}

// Tasks
export async function getTasks(): Promise<Task[]> {
  const { data, error } = await supabase
    .from('tasks_with_users')
    .select('*')
    .order('due_date', { ascending: true });
    
  if (error) {
    console.error('Error fetching tasks:', error);
    return [];
  }
  
  return data.map(task => {
    return {
      id: task.id,
      title: task.title,
      description: task.description || '',
      dueDate: task.due_date,
      priority: task.priority,
      assignedTo: {
        id: task.assigned_to || '',
        name: task.assignee_name || task.assignee_email?.split('@')[0] || 'Unassigned',
        email: task.assignee_email || '',
        avatar: task.assignee_avatar || '',
      },
      status: task.status,
      type: task.type,
      createdBy: {
        id: task.created_by,
        name: task.creator_name || task.creator_email?.split('@')[0] || 'Unknown',
      },
      relatedId: task.related_id || '',
      relatedType: task.related_type || '',
      createdAt: task.created_at,
      completedAt: task.completed_at,
    };
  });
}

export async function createTask(task: Partial<Task>): Promise<Task | null> {
  const user = await getCurrentUser();
  if (!user) return null;
  
  const { data, error } = await supabase
    .from('tasks')
    .insert({
      title: task.title || '',
      description: task.description,
      due_date: task.dueDate,
      priority: task.priority || 'medium',
      assigned_to: task.assignedTo?.id,
      status: task.status || 'pending',
      type: task.type || 'general',
      related_id: task.relatedId,
      related_type: task.relatedType,
      created_by: user.id,
    })
    .select()
    .single();
    
  if (error || !data) {
    console.error('Error creating task:', error);
    return null;
  }
  
  // Create an activity for this creation
  await supabase.from('activities').insert({
    type: 'task',
    description: `Task created: ${task.title}`,
    user_id: user.id,
    related_id: data.id, 
    related_type: 'task'
  });
  
  return {
    id: data.id,
    title: data.title,
    description: data.description || '',
    dueDate: data.due_date,
    priority: data.priority,
    assignedTo: {
      id: data.assigned_to || '',
      name: 'Assigned User', // This would need to be fetched separately 
      email: '',
      avatar: '',
    },
    status: data.status,
    type: data.type,
    createdBy: {
      id: data.created_by,
      name: user.email?.split('@')[0] || 'Current User',
    },
    relatedId: data.related_id || '',
    relatedType: data.related_type || '',
    createdAt: data.created_at,
    completedAt: data.completed_at,
  };
}

export async function updateTaskStatus(taskId: string, status: string): Promise<boolean> {
  const user = await getCurrentUser();
  if (!user) return false;
  
  const now = new Date().toISOString();
  const completedAt = status === 'completed' ? now : null;
  
  const { error } = await supabase
    .from('tasks')
    .update({ 
      status, 
      updated_at: now,
      completed_at: completedAt 
    })
    .eq('id', taskId);
    
  if (error) {
    console.error('Error updating task status:', error);
    return false;
  }
  
  return true;
}

// Notifications
export async function getNotifications() {
  const user = await getCurrentUser();
  if (!user) return [];
  
  const { data, error } = await supabase
    .from('notifications')
    .select('*')
    .eq('user_id', user.id)
    .eq('read', false)
    .order('created_at', { ascending: false });
    
  if (error) {
    console.error('Error fetching notifications:', error);
    return [];
  }
  
  return data;
}

export async function markNotificationAsRead(notificationId: string): Promise<boolean> {
  const { error } = await supabase
    .from('notifications')
    .update({ read: true })
    .eq('id', notificationId);
    
  if (error) {
    console.error('Error marking notification as read:', error);
    return false;
  }
  
  return true;
} 