import { createClient } from '@supabase/supabase-js'
import { createClientComponentClient } from '@supabase/auth-helpers-nextjs'
import { Database } from '@/types/supabase'

// These environment variables will need to be set in a .env.local file
// for local development or in the environment settings for production
const supabaseUrl = process.env.NEXT_PUBLIC_SUPABASE_URL || ''
const supabaseAnonKey = process.env.NEXT_PUBLIC_SUPABASE_ANON_KEY || ''

// Create a single supabase client for the entire app
export const supabase = createClient<Database>(supabaseUrl, supabaseAnonKey)

// Client component client for auth
export function getSupabaseClient() {
  return createClientComponentClient();
}

// Export types for future use with Supabase
export type SupabaseContact = {
  id: string
  name: string
  email: string | null
  phone: string | null
  company: string | null
  position: string | null
  last_activity: string | null
  status: 'active' | 'inactive'
  notes: string | null
  avatar: string | null
  tags: string[] | null
  created_at: string
  updated_at: string
}

// Client-side function to get authenticated user
export const getCurrentUser = async () => {
  const client = getSupabaseClient();
  const { data: { session } } = await client.auth.getSession();
  if (!session) return null;
  
  // Ensure user profile exists
  const user = session.user;
  await ensureUserProfile(user);
  
  return user;
};

// Helper function to ensure a user profile exists
export const ensureUserProfile = async (user: any) => {
  if (!user) return null;
  
  const client = getSupabaseClient();
  
  // Check if profile exists
  const { data: profile } = await client
    .from('profiles')
    .select('id')
    .eq('id', user.id)
    .single();

  if (profile) return profile;
  
  // Create profile if it doesn't exist
  try {
    const { error } = await client
      .from('profiles')
      .insert({ 
        id: user.id,
        name: user.user_metadata?.name || user.email?.split('@')[0] || 'User',
        email: user.email || '',
        role: 'user',
        created_at: new Date().toISOString(),
        updated_at: new Date().toISOString()
      });
    
    if (error) {
      console.error("Error creating profile:", error);
      return null;
    }

    return { id: user.id };
  } catch (err) {
    console.error("Exception creating profile:", err);
    return null;
  }
};

// Client-side helper function to get user profile
export const getUserProfile = async (userId: string) => {
  const client = getSupabaseClient();
  
  const { data, error } = await client
    .from('profiles')
    .select('*')
    .eq('id', userId)
    .single();
    
  if (error || !data) return null;
  
  return data;
}; 