-- 005_cart_setup.sql: Cart System Setup
BEGIN;

-- Cart table
CREATE TABLE IF NOT EXISTS public.cart (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    user_id UUID NOT NULL REFERENCES auth.users(id) ON DELETE CASCADE,
    product_id UUID NOT NULL REFERENCES public.content_items(id) ON DELETE CASCADE,
    quantity INTEGER NOT NULL DEFAULT 1 CHECK (quantity > 0),
    created_at TIMESTAMP WITH TIME ZONE DEFAULT now(),
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT now(),
    UNIQUE(user_id, product_id)
);

-- Indexes for performance
CREATE INDEX IF NOT EXISTS idx_cart_user_id ON public.cart(user_id);
CREATE INDEX IF NOT EXISTS idx_cart_product_id ON public.cart(product_id);

-- Enable Row Level Security
ALTER TABLE public.cart ENABLE ROW LEVEL SECURITY;

-- Policies
-- Allow users to view their own cart items
CREATE POLICY "Users can view their own cart items"
    ON public.cart
    FOR SELECT
    TO authenticated
    USING (auth.uid() = user_id);

-- Allow users to insert items into their own cart
CREATE POLICY "Users can add items to their own cart"
    ON public.cart
    FOR INSERT
    TO authenticated
    WITH CHECK (auth.uid() = user_id);

-- Allow users to update their own cart items
CREATE POLICY "Users can update their own cart items"
    ON public.cart
    FOR UPDATE
    TO authenticated
    USING (auth.uid() = user_id)
    WITH CHECK (auth.uid() = user_id);

-- Allow users to delete items from their own cart
CREATE POLICY "Users can delete their own cart items"
    ON public.cart
    FOR DELETE
    TO authenticated
    USING (auth.uid() = user_id);

-- Function to update updated_at timestamp
CREATE OR REPLACE FUNCTION public.handle_updated_at()
RETURNS TRIGGER AS $$
BEGIN
    NEW.updated_at = now();
    RETURN NEW;
END;
$$ language 'plpgsql';

-- Trigger to automatically update updated_at
CREATE TRIGGER handle_cart_updated_at
    BEFORE UPDATE ON public.cart
    FOR EACH ROW
    EXECUTE FUNCTION public.handle_updated_at();

-- Create explicit relationship for PostgREST
COMMENT ON COLUMN public.cart.product_id IS 'References the product in content_items table.';
COMMENT ON CONSTRAINT cart_product_id_fkey ON public.cart IS '@foreignKey (product_id) references content_items(id)';

COMMIT; 