"use client";
import { useState } from 'react';
import { getSupabaseClient } from '@/config/supabase';
import { useRouter } from 'next/navigation';
import Link from 'next/link';
import toast from 'react-hot-toast';

export default function LoginPage() {
  const router = useRouter();
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  async function handleLogin(e: React.FormEvent<HTMLFormElement>) {
    e.preventDefault();
    setError(null);
    setLoading(true);
    try {
      const supabase = getSupabaseClient();
      const { error } = await supabase.auth.signInWithPassword({ email, password });
      if (error) {
        setError(error.message);
        toast.error(error.message);
        return;
      }
      toast.success('Logged in successfully!');
      setTimeout(() => {
        router.replace('/account');
      }, 800);
    } catch (err) {
      setError('Unexpected error occurred.');
      toast.error('Unexpected error occurred.');
    } finally {
      setLoading(false);
    }
  }

  return (
    <>
      <h1 className="font-heading text-2xl font-semibold text-center mb-2">Sign in to your account</h1>
      <form className="flex flex-col gap-4" onSubmit={handleLogin} autoComplete="off">
        <label className="flex flex-col gap-2">
          <span className="font-medium text-foreground">Email</span>
          <input
            type="email"
            required
            autoComplete="email"
            className="input input-bordered rounded-lg px-4 py-2 bg-muted border border-muted-foreground focus:outline-none focus:ring-2 focus:ring-primary"
            value={email}
            onChange={e => setEmail(e.target.value)}
            placeholder="you@email.com"
          />
        </label>
        <label className="flex flex-col gap-2">
          <span className="font-medium text-foreground">Password</span>
          <input
            type="password"
            required
            autoComplete="current-password"
            className="input input-bordered rounded-lg px-4 py-2 bg-muted border border-muted-foreground focus:outline-none focus:ring-2 focus:ring-primary"
            value={password}
            onChange={e => setPassword(e.target.value)}
            placeholder="••••••••"
          />
        </label>
        {error && <div className="text-red-600 text-sm text-center">{error}</div>}
        <button
          className="bg-primary hover:bg-secondary text-primary-foreground font-semibold rounded-lg px-4 py-2 mt-2 shadow-card transition-all duration-150 disabled:opacity-70"
          type="submit"
          disabled={loading}
        >
          {loading ? 'Signing in...' : 'Sign in'}
        </button>
      </form>
      <div className="flex items-center justify-between mt-4 text-sm">
        <Link href="/auth/reset" className="text-primary hover:underline">Forgot password?</Link>
        <Link href="/auth/signup" className="text-primary hover:underline">Don't have an account?</Link>
      </div>
    </>
  );
}
