"use client";
import { useEffect, useState } from 'react';
import { getSupabaseClient } from '@/config/supabase';
import { useRouter, useSearchParams } from 'next/navigation';
import toast from 'react-hot-toast';

export default function ResetPasswordCallbackPage() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const [password, setPassword] = useState('');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    // If user visits reset page without required params, redirect
    if (!searchParams.has('access_token')) {
      router.replace('/auth/login');
    }
  }, [router, searchParams]);

  async function handleUpdatePassword(e: React.FormEvent<HTMLFormElement>) {
    e.preventDefault();
    setError(null);
    setLoading(true);
    try {
      const supabase = getSupabaseClient();
      const { error } = await supabase.auth.updateUser({ password });
      if (error) {
        setError(error.message);
        toast.error(error.message);
        return;
      }
      toast.success('Password updated! Please log in.');
      setTimeout(() => router.replace('/auth/login'), 1000);
    } catch (err) {
      setError('Unexpected error occurred.');
      toast.error('Unexpected error occurred.');
    } finally {
      setLoading(false);
    }
  }

  return (
    <>
      <h1 className="font-heading text-2xl font-semibold text-center mb-2">Set a new password</h1>
      <form className="flex flex-col gap-4" onSubmit={handleUpdatePassword} autoComplete="off">
        <label className="flex flex-col gap-2">
          <span className="font-medium text-foreground">New Password</span>
          <input
            type="password"
            required
            minLength={6}
            autoComplete="new-password"
            className="input input-bordered rounded-lg px-4 py-2 bg-muted border border-muted-foreground focus:outline-none focus:ring-2 focus:ring-primary"
            value={password}
            onChange={e => setPassword(e.target.value)}
            placeholder="At least 6 characters"
          />
        </label>
        {error && <div className="text-red-600 text-sm text-center">{error}</div>}
        <button
          className="bg-primary hover:bg-secondary text-primary-foreground font-semibold rounded-lg px-4 py-2 mt-2 shadow-card transition-all duration-150 disabled:opacity-70"
          type="submit"
          disabled={loading}
        >
          {loading ? 'Updating...' : 'Update password'}
        </button>
      </form>
    </>
  );
}
