"use client";
import { useState } from 'react';
import { getSupabaseClient } from '@/config/supabase';
import { useRouter } from 'next/navigation';
import Link from 'next/link';
import toast from 'react-hot-toast';

export default function SignupPage() {
  const router = useRouter();
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [username, setUsername] = useState('');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  async function handleSignup(e: React.FormEvent<HTMLFormElement>) {
    e.preventDefault();
    setError(null);
    setLoading(true);
    try {
      const supabase = getSupabaseClient();
      const { data, error: signUpError } = await supabase.auth.signUp({
        email,
        password
      });
      if (signUpError) {
        setError(signUpError.message);
        toast.error(signUpError.message);
        return;
      }
      // Set profile username
      if (data.user) {
        const { error: profileError } = await supabase
          .from('profiles')
          .insert({ 
            id: data.user.id,
            username,
            full_name: '',
            updated_at: new Date().toISOString()
          });
        if (profileError) {
          setError('Sign up successful, but failed to set profile.');
          toast.error('Sign up successful, but failed to set profile.');
        }
      }
      toast.success('Check your email to confirm your account!');
      setTimeout(() => {
        router.replace('/auth/login');
      }, 1200);
    } catch (err) {
      setError('Unexpected error occurred.');
      toast.error('Unexpected error occurred.');
    } finally {
      setLoading(false);
    }
  }

  return (
    <>
      <h1 className="font-heading text-2xl font-semibold text-center mb-2">Create an account</h1>
      <form className="flex flex-col gap-4" onSubmit={handleSignup} autoComplete="off">
        <label className="flex flex-col gap-2">
          <span className="font-medium text-foreground">Username</span>
          <input
            type="text"
            required
            minLength={3}
            maxLength={20}
            autoComplete="username"
            className="input input-bordered rounded-lg px-4 py-2 bg-muted border border-muted-foreground focus:outline-none focus:ring-2 focus:ring-primary"
            value={username}
            onChange={e => setUsername(e.target.value)}
            placeholder="yourname"
          />
        </label>
        <label className="flex flex-col gap-2">
          <span className="font-medium text-foreground">Email</span>
          <input
            type="email"
            required
            autoComplete="email"
            className="input input-bordered rounded-lg px-4 py-2 bg-muted border border-muted-foreground focus:outline-none focus:ring-2 focus:ring-primary"
            value={email}
            onChange={e => setEmail(e.target.value)}
            placeholder="you@email.com"
          />
        </label>
        <label className="flex flex-col gap-2">
          <span className="font-medium text-foreground">Password</span>
          <input
            type="password"
            required
            minLength={6}
            autoComplete="new-password"
            className="input input-bordered rounded-lg px-4 py-2 bg-muted border border-muted-foreground focus:outline-none focus:ring-2 focus:ring-primary"
            value={password}
            onChange={e => setPassword(e.target.value)}
            placeholder="At least 6 characters"
          />
        </label>
        {error && <div className="text-red-600 text-sm text-center">{error}</div>}
        <button
          className="bg-primary hover:bg-secondary text-primary-foreground font-semibold rounded-lg px-4 py-2 mt-2 shadow-card transition-all duration-150 disabled:opacity-70"
          type="submit"
          disabled={loading}
        >
          {loading ? 'Creating account...' : 'Create account'}
        </button>
      </form>
      <div className="flex justify-between mt-4 text-sm">
        <Link href="/auth/login" className="text-primary hover:underline">Already have an account?</Link>
      </div>
    </>
  );
}
