"use client";
import { useCartContext } from '@/features/cart/CartContext';
import Image from 'next/image';
import { formatPrice } from '@/utils/format';

export default function CartPage() {
  const {
    cart,
    loading,
    error,
    updateQuantity,
    removeFromCart,
    clearCart,
    subtotal
  } = useCartContext();

  if (loading)
    return (
      <section className="max-w-3xl mx-auto px-4 py-12 min-h-[300px] flex flex-col items-center justify-center">
        <h1 className="text-3xl font-heading font-bold mb-6">Your Cart</h1>
        <div className="text-muted-foreground text-lg mt-6 animate-pulse">Loading cart...</div>
      </section>
    );

  if (error)
    return (
      <section className="max-w-3xl mx-auto px-4 py-12 min-h-[300px] flex flex-col items-center justify-center">
        <h1 className="text-3xl font-heading font-bold mb-6">Your Cart</h1>
        <div className="text-red-600 text-lg mt-6">{error}</div>
      </section>
    );

  if (!cart.length)
    return (
      <section className="max-w-3xl mx-auto px-4 py-12 min-h-[300px] flex flex-col items-center justify-center">
        <h1 className="text-3xl font-heading font-bold mb-6">Your Cart</h1>
        <div className="text-muted-foreground text-lg mt-6">Your cart is empty.</div>
      </section>
    );

  return (
    <section className="max-w-3xl mx-auto px-4 py-12 min-h-[300px] flex flex-col items-center">
      <h1 className="text-3xl font-heading font-bold mb-6">Your Cart</h1>
      <div className="w-full bg-background/90 rounded-xl shadow-card border border-muted divide-y divide-muted mb-6">
        {cart.map(item => (
          <div key={item.id} className="flex flex-col sm:flex-row gap-4 items-center px-4 py-4">
            <div className="w-20 h-20 flex-shrink-0 rounded-xl overflow-hidden border border-muted bg-muted">
              {item.product && (
                <Image
                  src={item.product.image_url}
                  alt={item.product.title}
                  width={80}
                  height={80}
                  className="object-cover h-full w-full"
                />
              )}
            </div>
            <div className="flex-1 flex flex-col gap-1">
              <div className="font-heading font-semibold text-base truncate">
                {item.product?.title}
              </div>
              <div className="text-muted-foreground text-sm line-clamp-2">
                {item.product?.description}
              </div>
              <div className="font-bold text-primary mt-1">
                {item.product ? formatPrice(item.product.price) : null}
              </div>
            </div>
            <div className="flex flex-col items-center gap-2 min-w-[90px]">
              <label htmlFor={`qty-${item.id}`} className="sr-only">Quantity</label>
              <input
                id={`qty-${item.id}`}
                type="number"
                min={1}
                value={item.quantity}
                onChange={e => updateQuantity(item.id, Math.max(1, Number(e.target.value)))}
                className="w-16 px-2 py-1 rounded border border-muted-foreground bg-muted focus:ring-2 focus:ring-primary text-center font-medium"
              />
              <button
                className="text-xs text-red-600 hover:underline mt-1"
                type="button"
                onClick={() => removeFromCart(item.id)}
              >
                Remove
              </button>
            </div>
          </div>
        ))}
      </div>
      <div className="flex justify-between w-full max-w-md mb-8">
        <div className="font-semibold text-lg">Subtotal</div>
        <div className="font-bold text-primary text-lg">{formatPrice(subtotal)}</div>
      </div>
      <div className="flex gap-3 w-full max-w-md">
        <button
          className="flex-1 px-6 py-2 rounded-lg bg-muted text-foreground font-semibold border border-muted hover:bg-background transition-colors"
          onClick={clearCart}
        >
          Clear Cart
        </button>
        <button
          className="flex-1 px-6 py-2 rounded-lg bg-primary text-primary-foreground font-semibold shadow-card hover:bg-secondary hover:scale-105 transition-all duration-150"
          disabled
        >
          Checkout (Coming Soon)
        </button>
      </div>
    </section>
  );
}
