"use client";
import Link from 'next/link';
import { useState, useEffect } from 'react';
import { ShoppingCartIcon } from '@heroicons/react/24/outline';
import clsx from 'clsx';
import dynamic from 'next/dynamic';
import { useCartContext } from '@/features/cart/CartContext';

const UserMenu = dynamic(() => import('./UserMenu'), { ssr: false });

export default function Navbar() {
  const [scrolled, setScrolled] = useState(false);
  const [prevCartCount, setPrevCartCount] = useState(0);
  const [animateCount, setAnimateCount] = useState(false);
  
  useEffect(() => {
    const onScroll = () => setScrolled(window.scrollY > 8);
    window.addEventListener('scroll', onScroll);
    return () => window.removeEventListener('scroll', onScroll);
  }, []);

  const { cart, loading: cartLoading } = useCartContext();
  const cartCount = cart.reduce((sum, item) => sum + item.quantity, 0);

  // Handle cart count changes
  useEffect(() => {
    if (cartCount !== prevCartCount) {
      setPrevCartCount(cartCount);
      setAnimateCount(true);
      const timer = setTimeout(() => setAnimateCount(false), 300);
      return () => clearTimeout(timer);
    }
  }, [cartCount, prevCartCount]);

  return (
    <header
      className={clsx(
        'sticky top-0 z-40 w-full transition-shadow duration-300',
        scrolled ? 'shadow-glass bg-background/80 backdrop-blur' : 'bg-transparent'
      )}
    >
      <nav className="max-w-7xl mx-auto px-4 py-3 flex items-center justify-between">
        <Link href="/" className="flex items-center gap-2 font-heading text-2xl font-bold">
          <span className="bg-gradient-to-r from-primary to-secondary bg-clip-text text-transparent">
            Amazona
          </span>
        </Link>
        <ul className="flex items-center gap-6">
          <li>
            <Link
              href="/"
              className="hover:text-primary transition-colors font-medium"
            >Home</Link>
          </li>
          <li>
            <Link
              href="/cart"
              className="relative group flex items-center hover:text-primary transition-colors font-medium"
            >
              <ShoppingCartIcon className="w-6 h-6" />
              <span className="sr-only">Cart</span>
              <span 
                key={cartCount}
                className={clsx(
                  "absolute -top-2 -right-2 bg-accent text-xs rounded-full px-1.5 py-0.5 text-white shadow-card transition-all duration-300",
                  animateCount ? "scale-125" : "group-hover:scale-105",
                  cartCount === 0 ? "opacity-0" : "opacity-100"
                )}
              >
                {cartLoading ? (
                  <span className="animate-pulse">...</span>
                ) : (
                  <span className="min-w-[1ch] inline-block text-center">
                    {cartCount}
                  </span>
                )}
              </span>
            </Link>
          </li>
          <li>
            <UserMenu />
          </li>
        </ul>
      </nav>
    </header>
  );
}
